package com.ovopark.iohub.sdk.client.instream;

import com.ovopark.iohub.sdk.client.outstream.RequestParamBody;
import com.ovopark.iohub.sdk.model.proto.ImportPreDefConf;
import com.ovopark.iohub.sdk.model.proto.LimitLogger;
import com.ovopark.iohub.sdk.model.proto.NoPrivilegeException;
import com.ovopark.iohub.sdk.model.proto.Segment;
import com.ovopark.kernel.shared.Model;
import com.ovopark.module.shared.Session;
import lombok.Data;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.Map;

public interface JobInTaskFlow<T extends RequestParamBody> {

    default void prepared(){};

    /**
     * <PRE>
     * provide request parameter for exporting again ,
     * the param is also stored in export hub of organize
     *
     * throw {@link NoPrivilegeException} if the report is rejected , does not match some condition
     *
     * @param requestBody http post request body / json object data format
     * @param session authentication
     * @return T
     * @throws NoPrivilegeException no privilege
     */
    T requestParamBody(String requestBody, Session session) throws NoPrivilegeException;


    default ImportTaskInfoProvider importTaskInfoProvider(T requestParamBody, Session session){return null;}

    void execute(DataRowStream<T> dataRowStream);

    default void close(){};

    default Class<T> paramType(){
        Type superClass = this.getClass().getGenericInterfaces()[0];
        if (superClass instanceof Class) {
            throw new IllegalArgumentException("Internal error: TypeReference constructed without actual type information");
        } else {
            return (Class<T>) ((ParameterizedType)this.getClass().getGenericInterfaces()[0]).getActualTypeArguments()[0];
        }
    }

    interface ImportTaskInfoProvider {

        String name();

    }

    interface HeaderLookup{

        Segment.Header.Cell lookup(String path);

    }

    @Data
    class DataRowImpl implements DataRow{

        private int segment;

        private int rowNum;

        Map<String, Object> data;

        transient private HeaderLookup headerLookup;

        @Override
        public int segment() {
            return segment;
        }

        @Override
        public int rowNum() {
            return rowNum;
        }

        @Override
        public Map<String, Object> data() {
            return data;
        }

        @Override
        public Segment.Header.Cell header(String path) {
            return headerLookup.lookup(path);
        }
    }

    interface DataRowStream<T extends RequestParamBody> {

        /**
         * reset stream ,then subscribe this again.
         * @return
         */
        DataRowStream<T> reset();

        void subscribe(DataRowListener dataRowListener);

        int segmentCount();

        long rowCount();

        long rowCount(int segment);

        Session session();

        ImportConf<T> importConf();

        void mark(StreamProcessResult streamProcessResult, String msg);

        /**
         * <PRE>
         * 这个日志可以把内容推送到control， 最大只能推送 1W 次，
         * 尽量在关键性的节点打印日志
         *
         * 日志也会通过Slf4j打印 , info级别
         *
         * 支持并发
         * @return
         */
        LimitLogger limitLogger();

    }

    interface DataRowListener{

        void onRow(DataRow dataRow, ImportContext importContext);

        default void onHeader(int segment,Segment.Header header, ImportContext importContext){};

    }

    interface ImportContext {

        void cancel();

        default void reply(int fromRow, int toRow,String desc,Result result){
            replyWithSegment(0,fromRow,toRow,desc,result);
        }

        default void reply( int row,String desc,Result result){
            reply(row,row,desc,result);
        }

        default void replyWithSegment(int segment, int row, String desc,Result result){
            replyWithSegment(segment,row,row,desc,result);
        }

        void replyWithSegment(int segment, int fromRow, int toRow, String desc,Result result);
    }

    interface DataRow {

        int segment();

        int rowNum();

        Map<String, Object> data();

        Segment.Header.Cell header(String path);

    }

    interface ImportConf<T extends RequestParamBody> extends Model {

        T requestParam();

        ImportPreDefConf importPreDefConf();


    }

    enum Result{

        SUCCESS, FAIL , EXISTS

    }

    enum StreamProcessResult {

        SUCCESS, FAIL

    }

    interface RuntimeStat{

        void rowCountAdd(int added);

        int rowCount();

        void currentSegment(String segment);

        String currentSegment();

        boolean cancelled();

        void cancel();

    }
}
