package com.ovopark.iohub.sdk.model.proto.internal;

import com.ovopark.iohub.sdk.model.proto.OutStore;
import com.ovopark.iohub.sdk.model.proto.Segment;
import com.ovopark.kernel.shared.DBOpeException;
import com.ovopark.kernel.shared.JSONAccessor;
import com.ovopark.kernel.shared.vfile.AppendLog;
import lombok.extern.slf4j.Slf4j;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import static com.ovopark.kernel.shared.Util.compress;

@Slf4j
public class MixedRowTransLogImpl implements RowTransLog {

    final private AppendLog appendLog;

    private Segment.RowRegion rowRegion;

    private final RowTransLogConfig rowTransLogConfig;

    final String filePath;

    private long byteSize;

    final OutStore.RuntimeStat runtimeStat;

    public MixedRowTransLogImpl(String tag, String filePath, RowTransLogConfig rowTransLogConfig,OutStore.RuntimeStat runtimeStat) {

        AppendLog.AppendLogConf appendLogConf=new AppendLog.AppendLogConf();
        appendLogConf.setWalBufferSizeMb(rowTransLogConfig.getWalBufferSizeMb());
        appendLogConf.setWalDiskSizeMb(rowTransLogConfig.getWalDiskSizeMb());
        appendLogConf.setWalIntervalSec(rowTransLogConfig.getWalIntervalSec());
        appendLogConf.setCompressed(rowTransLogConfig.isCompressed());
        appendLog =new AppendLog.BufferedAppendLogImpl(tag,filePath, appendLogConf);
        rowRegion=new Segment.RowRegion();
        this.rowTransLogConfig=rowTransLogConfig;
        this.filePath=filePath;
        this.runtimeStat=runtimeStat;
    }


    @Override
    public synchronized void append(Map<String, Object> data) {
        rowRegion.getDataList().add(data);
        fsyncRegion(false);
    }

    private void fsyncRegion(boolean force){
        List<Map<String, Object>> dataList = rowRegion.getDataList();
        if (dataList.size()>rowTransLogConfig.getRegionRowCount() || force) {
            //commit
            try {
                byte[] data = JSONAccessor.impl().formatAsBytes(rowRegion);
                runtimeStat.maxRowRegionByteSize(data.length);
                runtimeStat.jsonByteSizeAdd(data.length);
                if (rowTransLogConfig.isCompressed()) {
                    data=compress(data);
                    runtimeStat.compressedByteSizeAdd(data.length);
                }
                long append = appendLog.append(data);
                byteSize+=append;
            } catch (Exception e) {
                log.error(rowRegion.getDataList().size()+" / " +rowTransLogConfig.getRegionRowCount()
                                +" rows, size exceed "+rowTransLogConfig.getWalBufferSizeMb()+"MB???? ");
                throw DBOpeException.from(e);
            }
            if (appendLog.freeSize4G()<0) {
                throw new IllegalStateException("error, exceed max file size: "+rowTransLogConfig.getWalDiskSizeMb()+" MB, file: "+filePath);
            }
            rowRegion=new Segment.RowRegion();
        }
    }

    @Override
    public synchronized void append(List<Map<String, Object>> data) {
        rowRegion.getDataList().addAll(data);
        fsyncRegion(false);
    }

    @Override
    public synchronized void fsync() throws DBOpeException {
        //fsync region
        fsyncRegion(true);
        appendLog.fsync();
    }


    @Override
    public synchronized void close() throws IOException {
        try {
            fsync();
        }
        finally {
            appendLog.close();
        }
    }

    @Override
    public long byteSize() {
        return byteSize;
    }

    @Override
    public List<Map<String, Object>> capture(int n) {
        List<Map<String, Object>> rowList = rowRegion.getDataList();
        int size = rowList.size();
        if (size==0) {
            return new ArrayList<>(0);
        }
        int f = size - n;
        if (f<0) {
            f=0;
        }
        List<Map<String,Object>> captureList=new ArrayList<>(Math.min(n, size));
        for (int i = f; i < size; i++) {
            captureList.add(rowList.get(i));
        }
        return captureList;
    }
}
