package com.ovopark.iohub.sdk.model.proto.internal;

import com.ovopark.iohub.sdk.model.proto.CellFile;
import com.ovopark.iohub.sdk.model.proto.LimitLogger;
import com.ovopark.iohub.sdk.model.proto.OutStore;
import com.ovopark.iohub.sdk.model.proto.Segment;
import com.ovopark.kernel.shared.JSONAccessor;
import com.ovopark.kernel.shared.Model;
import lombok.Getter;

import java.util.*;
import java.util.concurrent.CancellationException;

import static com.ovopark.kernel.shared.Util.isEmpty;

public class InMemoryOutStore implements OutStore, LimitLogger.LimitLoggerSetter {

    @Getter
    private String fileName;

    private final List<Segment> segmentList;

    private boolean committed;

    final Stat stat=new Stat();

    final RuntimeStat runtimeStat=new RuntimeStatImpl();

    @Getter
    final private Feature feature=new Feature();

    @Getter
    final private Map<String,Group> groupMap=new HashMap<>();

    LimitLogger limitLogger;

    final int maxRowCountInMemory;

    public InMemoryOutStore(int size) {
        this(size,10_000);
    }

    public InMemoryOutStore(int size,int maxRowCountInMemory) {
        this.segmentList = new ArrayList<>(size);
        this.maxRowCountInMemory=maxRowCountInMemory;
    }

    @Override
    public SegmentWriter createSegment(String name) {
        return createSegment(name,10);
    }

    @Override
    public SegmentWriter createSegment(String name,int rowSize) {
        return createSegment(name,rowSize,"default");
    }

    @Override
    public SegmentWriter createSegment(String name, String group) {
        return createSegment(name,10,group);
    }

    @Override
    public SegmentWriter createSegment(String name, int rowSize, String group) {
        check();
        if (isEmpty(group)) {
            throw new IllegalArgumentException("group is empty");
        }
        groupMap.putIfAbsent(group,new Group());
        SegmentWriterImpl segmentWriter = new SegmentWriterImpl(name, rowSize, group);
        runtimeStat.segmentAdd(group+":"+name);
        runtimeStat.segmentCountAdd(1);
        limitLogger.log("create segment "+group+":"+name);
        return segmentWriter;
    }

    @Override
    public SegmentWriter cloneSegmentExcludeRowData(String name, String target) {
        Segment sf=null;
        for (Segment segment : segmentList) {
            if (segment.name().equals(name)) {
                sf=segment;
                break;
            }
        }

        if (sf==null) {
            throw new IllegalArgumentException("cannot find segment: "+name);
        }

        SegmentImpl sfi = (SegmentImpl) sf;

        SegmentWriter segmentWriter = createSegment(target, 8, sfi.getGroup());
        Segment.Header targetHeader = segmentWriter.header();

        sfi.header().scanUp2Down(new Segment.Header.Scan() {
            @Override
            public void scan(Segment.Header.Cell cell) {
                String p = cell.path();
                if (p.contains(".")) {
                    targetHeader.upset(p.substring(0,p.lastIndexOf(".")),p.substring(p.lastIndexOf(".")),null);
                }
                else {
                    targetHeader.upset(null,p,null);
                }
                HeaderCellImpl c = (HeaderCellImpl) cell;
                // feature
                HeaderCellImpl targetCell = ((HeaderCellImpl) targetHeader.cell(p));

                targetCell.setWidth(c.getWidth());
                targetCell.setI18Key(c.getI18Key());
                targetCell.setName(c.getName());
                targetCell.setTip(c.getTip());

                targetCell.setVirtual(c.isVirtual());

                Segment.HeaderFeature feature1 = c.feature();
                if (feature1 != null) {
                    Segment.HeaderFeature headerFeature = feature1.clone();
                    targetCell.setHeaderFeature(headerFeature);
                }
            }
        });
        segmentWriter.title().title(sfi.title().title());
        return segmentWriter;
    }

    @Override
    public void commit() {
        this.committed=true;
    }

    @Override
    public void close() {
        for (Segment segment : segmentList) {
            segment.commit();
        }
    }

    @Override
    public Segment segment(Segment.SD sd) {
        for (Segment segment : segmentList) {
            if (segment.name().equals(sd.getName()) && segment.group().equals(sd.getGroup())) {
                return segment;
            }
        }
        return null;
    }

    @Override
    public void fileName(String fileName) {
        this.fileName=fileName;
    }

    @Override
    public GroupOperation group(String group) {
        return new GroupOperation() {
            @Override
            public void fileName(String fileName) {
                groupMap.get(group).setFileName(fileName);
            }
        };
    }

    @Override
    public FeatureOperation feature() {
        return new FeatureOperation() {
            @Override
            public void supportRangeMerge(boolean rangeMerge) {
                feature.setRangeMerge(rangeMerge);
            }

            @Override
            public void supportBorder(boolean border) {
                feature.setBorder(border);
            }

            @Override
            public void supportParallelism(boolean parallelism) {
                feature.setParallelism(parallelism);
            }
        };
    }

    @Override
    public Stat stat() {
        return stat;
    }

    @Override
    public RuntimeStat runtimeStat() {
        return runtimeStat;
    }

    @Override
    public List<Segment.SD> sdList() {
        List<Segment.SD> sdList=new ArrayList<>(segmentList.size());
        for (Segment segment : segmentList) {
            Segment.SD sd=new Segment.SD();
            sd.setMemory(true);
            sd.setName(segment.name());
            sd.setGroup(segment.group());
            sdList.add(sd);
        }
        return sdList;
    }

    @Override
    public CellFile file(byte[] data, String name, String type) {
        CellFileImpl cellFile=new CellFileImpl();
        cellFile.setName(name);
        cellFile.setType(type);
        cellFile.setBase64(true);
        cellFile.setUri(Base64.getEncoder().encodeToString(data));
        return cellFile;
    }

    @Override
    public LimitLogger limitLogger() {
        return limitLogger;
    }

    @Override
    public Map<String, List<Map<String, Object>>> capture(int n) {
        Map<String, List<Map<String,Object>>> listMap=new LinkedHashMap<>();
        int size = segmentList.size();
        for (int i = 0; i < size; i++) {
            Segment segment = segmentList.get(i);
            List<Map<String, Object>> list = segment.capture(n);
            listMap.put(segment.name(),list);
        }
        return listMap;
    }

    private void check(){
        if (committed) {
            throw new UnsupportedOperationException("all committed???");
        }
    }

    @Override
    public void setLimitLogger(LimitLogger limitLogger) {
        this.limitLogger=limitLogger;
    }

    private class SegmentWriterImpl implements SegmentWriter{


        private final SegmentImpl segment;

        private boolean committed;

        final private SegmentStat segmentStat=new SegmentStat();

        public SegmentWriterImpl(String name,int size,String group) {
            this.segment = new SegmentImpl(name,size,group);
        }

        @Override
        public synchronized void append(Map<String, Object> data) {
            check();
            segment.append(data);
            runtimeStat.rowCountAdd(1);
            runtimeStat.lastIoTimeMs(System.currentTimeMillis());
            if (segment.size()>maxRowCountInMemory) {
                throw new IllegalStateException("too many row: "+segment.size()+", max: "+maxRowCountInMemory);
            }
        }

        @Override
        public synchronized void append(List<Map<String, Object>> data) {
            check();
            for (Map<String, Object> datum : data) {
                segment.append(datum);
            }
            runtimeStat.rowCountAdd(data.size());
            runtimeStat.lastIoTimeMs(System.currentTimeMillis());
            if (segment.size()>maxRowCountInMemory) {
                throw new IllegalStateException("too many row: "+segment.size()+", max: "+maxRowCountInMemory);
            }
        }

        @Override
        public void meta(Map<String, Object> meta) {
            segment.setMeta(meta);
        }

        @Override
        public synchronized void append(Model model) {
            append(JSONAccessor.impl().read(JSONAccessor.impl().format(model)));
        }

        @Override
        public synchronized void commit() {
            if (committed) {
                return;
            }
            segmentList.add(segment);
            segment.header().commit();
            segment.commit();

            segmentStat.setRowCount(segment.getRowList().size());
            if (stat.getSegmentStatList()==null) {
                stat.setSegmentStatList(new ArrayList<>());
            }
            stat.getSegmentStatList().add(segmentStat);

            this.committed=true;
        }

        @Override
        public Segment.Title title() {
            return segment.getTitle();
        }


        private void check(){
            if (committed) {
                throw new UnsupportedOperationException("segment committed???");
            }
            if (runtimeStat.cancelled()) {
                throw new CancellationException("task is cancelled");
            }
        }

        @Override
        public Segment.Header header() {
            return segment.header();
        }

        @Override
        public String group() {
            return segment.group();
        }

        @Override
        public FeatureOperation feature() {
            return new FeatureOperation() {
                @Override
                public void supportRangeMerge(boolean rangeMerge) {
                    segment.feature().setRangeMerge(rangeMerge);
                }

                @Override
                public void supportBorder(boolean border) {
                    segment.feature().setBorder(border);
                }

                @Override
                public void defaultValIfNullOrEmpty(String val) {
                    segment.feature().setDefaultValIfNullOrEmpty(val);
                }
            };
        }

        @Override
        public List<Map<String,Object>> capture(int n) {
            return segment.capture(n);
        }
    }

}
