package com.ovopark.iohub.sdk.client;

import com.ovopark.iohub.sdk.model.*;
import com.ovopark.iohub.sdk.model.instream.ReadJobAssignWorkRequest;
import com.ovopark.iohub.sdk.model.instream.ReadJobAssignWorkResponse;
import com.ovopark.iohub.sdk.model.instream.ReadJobStartRequest;
import com.ovopark.iohub.sdk.model.instream.ReadJobStartResponse;
import com.ovopark.iohub.sdk.model.outstream.RenderJobAssignWorkRequest;
import com.ovopark.iohub.sdk.model.outstream.RenderJobAssignWorkResponse;
import com.ovopark.iohub.sdk.model.outstream.RenderJobStartRequest;
import com.ovopark.iohub.sdk.model.outstream.RenderJobStartResponse;
import com.ovopark.iohub.sdk.model.proto.TaskCreatedRequest;
import com.ovopark.kernel.shared.Config;
import com.ovopark.kernel.shared.JSONAccessor;
import com.ovopark.module.shared.BaseResult;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.io.FileSystemResource;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.ResponseEntity;
import org.springframework.http.client.SimpleClientHttpRequestFactory;
import org.springframework.http.converter.StringHttpMessageConverter;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestTemplate;

import java.nio.charset.StandardCharsets;
import java.util.Map;

@Slf4j
public class Client2ControlRestClient implements Client2ControlTransport {


    final boolean verbose= Config.ConfigPriority.option().getBoolean("iohub.client.io.verbose",true);

    private final RestTemplate restTemplate;

    final private ControlNode controlNode;

    public Client2ControlRestClient(ControlNode controlNode,RestTemplate restTemplate) {
        this.controlNode=controlNode;
        this.restTemplate=restTemplate;
    }

    public Client2ControlRestClient(ControlNode controlNode) {
        this.controlNode = controlNode;
        SimpleClientHttpRequestFactory factory = new SimpleClientHttpRequestFactory();
        factory.setReadTimeout(45_000);//ms
        factory.setConnectTimeout(15_000);//ms
        restTemplate = new RestTemplate(factory);
        restTemplate.getMessageConverters().add(0,new StringHttpMessageConverter(StandardCharsets.UTF_8));
    }



    @Override
    public ClientNodeRegisterResponse heartbeat(ClientNodeRegisterRequest clientNodeRegisterRequest) {
        return rpc(clientNodeRegisterRequest, "/jobClient/heartbeat",ClientNodeRegisterResponse.class);
    }

    private <T> T rpc(Object request, String path,Class<T> clazz) {
        String url = "http://"+ controlNode.ip() + ":" + controlNode.port()+"/iohub-control" ;
        url+=path;
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type","application/json");
        HttpEntity<String> formEntity = new HttpEntity<String>(JSONAccessor.impl().format(request), headers);
        try {
            ResponseEntity<String> responseEntity = restTemplate.postForEntity(url, formEntity, String.class);
            int statusCodeValue = responseEntity.getStatusCodeValue();
            if (statusCodeValue==200) {
                return JSONAccessor.impl().read(responseEntity.getBody(), clazz);
            }
            return null;
        } catch (Exception e) {
            if (verbose) {
                log.error(e.getMessage());
            }
            return null;
        }
    }

    private <T> T restWithFile(Map<String,Object> params, String file,  String path, Class<T> clazz){
        String url = "http://"+ controlNode.ip() + ":" + controlNode.port()+"/iohub-control" ;
        url+=path;
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type","multipart/form-data");
        MultiValueMap<String, Object> multiValueMap = new LinkedMultiValueMap<>();
        for (Map.Entry<String, Object> entry : params.entrySet()) {
            multiValueMap.add(entry.getKey(),entry.getValue());
        }
        multiValueMap.add("file",new FileSystemResource(file));

        HttpEntity<MultiValueMap<String, Object>> formEntity = new HttpEntity<>(multiValueMap, headers);
        try {
            ResponseEntity<String> responseEntity = restTemplate.postForEntity(url, formEntity, String.class);
            int statusCodeValue = responseEntity.getStatusCodeValue();
            if (statusCodeValue==200) {
                return JSONAccessor.impl().read(responseEntity.getBody(), clazz);
            }
            return null;
        } catch (Exception e) {
            log.error(e.getMessage());
            return null;
        }

    }

    @Override
    public TaskLockResponse lockClient(TaskLockRequest taskLockRequest) {
        return rpc(taskLockRequest, "/jobClient/lockClient",TaskLockResponse.class);
    }

    @Override
    public TaskLogResponse log(TaskLogRequest taskLogRequest) {
        return rpc(taskLogRequest, "/jobClient/log",TaskLogResponse.class);
    }

    @Override
    public PreFlowErrorResponse preFlowError(PreFlowErrorRequest preFlowErrorRequest) {
        return rpc(preFlowErrorRequest, "/jobClient/preFlowError", PreFlowErrorResponse.class);
    }

    @Override
    public RenderJobStartResponse startRenderJob(RenderJobStartRequest renderJobStartRequest) {
        return rpc(renderJobStartRequest, "/jobClient/renderJob/start", RenderJobStartResponse.class);
    }


    @Override
    public RenderJobAssignWorkResponse assignWorkRenderJob(RenderJobAssignWorkRequest renderJobAssignWorkRequest) {
        return rpc(renderJobAssignWorkRequest, "/jobClient/renderJob/assignWork", RenderJobAssignWorkResponse.class);
    }


    @Override
    public ReadJobStartResponse startReadJob(ReadJobStartRequest readJobStartRequest) {
        return rpc(readJobStartRequest, "/jobClient/readJob/start", ReadJobStartResponse.class);
    }


    @Override
    public ReadJobAssignWorkResponse assignWorkReadJob(ReadJobAssignWorkRequest readJobAssignWorkRequest) {
        return rpc(readJobAssignWorkRequest, "/jobClient/readJob/assignWork", ReadJobAssignWorkResponse.class);
    }

    @Override
    public BaseResult<?> watermarkSync(TaskCreatedRequest taskCreatedRequest) {
        return rpc(taskCreatedRequest, "/export/watermarkSync", BaseResult.class);
    }

}
