package com.ovopark.jobhub.sdk.client.kafka;

import com.ovopark.jobhub.sdk.client.*;
import com.ovopark.jobhub.sdk.model.internal.TaskLockRequest;
import com.ovopark.jobhub.sdk.model.internal.TaskLockResponse;
import com.ovopark.jobhub.sdk.model.internal.TaskSubmitRequest;
import com.ovopark.kernel.shared.JSONAccessor;
import com.ovopark.kernel.shared.ShutdownManager;
import com.ovopark.kernel.shared.Util;
import com.ovopark.kernel.shared.vclient.ClientNode;
import lombok.extern.slf4j.Slf4j;
import org.apache.kafka.clients.consumer.ConsumerConfig;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.CommandLineRunner;
import org.springframework.kafka.core.ConsumerFactory;
import org.springframework.kafka.listener.AcknowledgingMessageListener;
import org.springframework.kafka.listener.ConcurrentMessageListenerContainer;
import org.springframework.kafka.listener.ContainerProperties;
import org.springframework.kafka.support.Acknowledgment;
import org.springframework.stereotype.Component;

import java.time.LocalDateTime;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;

import static com.ovopark.kernel.shared.Util.*;

@JobClientActive
@Component
@Slf4j
public class JobListenerRunnerViaKafkaProvider implements JobService.JobListenerKafkaRunnerProvider, CommandLineRunner {

    @Autowired
    private DelayTaskExecutor delayTaskExecutor;

    @Autowired
    private Client2ControlTransport client2ControlTransport;

    @Autowired
    private JobService jobService;

    @Qualifier("com.ovopark.jobhub.sdk.client.kafka.KafkaConfig.consumerFactory")
    @Autowired
    ConsumerFactory<String, String> consumerFactory;

    private final Map<String, ConcurrentMessageListenerContainer<String, String>> containers = new ConcurrentHashMap<>();

    @Autowired
    JobInnerContextMgr jobInnerContextMgr;

    @Override
    public String name() {
        return "kafka";
    }

    @Override
    public boolean start(final String topic,String beanUrl,JobService.JobListenerRunnerKafkaConfig jobListenerRunnerKafkaConfig) {
        final String containerId=beanUrl;
        ConcurrentMessageListenerContainer<String, String> container = lock(containerId, new Callable<ConcurrentMessageListenerContainer<String, String>>() {
            @Override
            public ConcurrentMessageListenerContainer<String, String> call() throws Exception {

                ConcurrentMessageListenerContainer<String, String> listenerContainer = containers.get(containerId);
                if (listenerContainer!=null) {
                    return listenerContainer;
                }

                final String groupId = isEmpty(jobListenerRunnerKafkaConfig.getConsumerGroup())?containerId:
                        jobListenerRunnerKafkaConfig.getConsumerGroup();
                // 创建容器
                ContainerProperties containerProps = new ContainerProperties(topic);
                containerProps.setGroupId(groupId);
                // 通过 consumer properties 覆盖 offset reset
                if (isNotEmpty(jobListenerRunnerKafkaConfig.getOffsetReset())) {
                    containerProps.getKafkaConsumerProperties()
                            .put(ConsumerConfig.AUTO_OFFSET_RESET_CONFIG, jobListenerRunnerKafkaConfig.getOffsetReset()); // e.g., "earliest"
                }

                final boolean autoCommit = jobListenerRunnerKafkaConfig.isAutCommit();
                containerProps.getKafkaConsumerProperties().put(ConsumerConfig.ENABLE_AUTO_COMMIT_CONFIG, autoCommit);
                if (!autoCommit) {
                    // 👇 关键：设置为 MANUAL 或 MANUAL_IMMEDIATE
                    containerProps.setAckMode(ContainerProperties.AckMode.MANUAL);
                }
                int pollIntervalTimeSec = jobListenerRunnerKafkaConfig.getPollIntervalTimeSec();
                if (pollIntervalTimeSec>0) {
                    containerProps.getKafkaConsumerProperties().put(ConsumerConfig.MAX_POLL_INTERVAL_MS_CONFIG, pollIntervalTimeSec*1000);
                }
                containerProps.getKafkaConsumerProperties().put(ConsumerConfig.PARTITION_ASSIGNMENT_STRATEGY_CONFIG
                        , "org.apache.kafka.clients.consumer.CooperativeStickyAssignor");

                containerProps.setMessageListener(new AcknowledgingMessageListener<String, String>() {
                    @Override
                    public void onMessage(ConsumerRecord<String, String> record, Acknowledgment acknowledgment) {
                        String value = record.value();
                        TaskSubmitRequest taskSubmitRequest = JSONAccessor.impl().read(value, TaskSubmitRequest.class);
                        final long cronTaskId = taskSubmitRequest.getCronTaskId();
                        final long cronTaskHistoryId = taskSubmitRequest.getCronTaskHistoryId();
                        final String taskKey = "task4kafka_"+jobInnerContextMgr.key(cronTaskId,cronTaskHistoryId);
                        MDC.put("traceId", taskKey);
                        MDC.put("requestId", taskKey);
                        JobLog jobLog=new JobLogImpl(client2ControlTransport,taskKey, cronTaskHistoryId);
                        try {
                            TaskLockRequest taskLockRequest=new TaskLockRequest();
                            taskLockRequest.setCronTaskId(cronTaskId);
                            taskLockRequest.setCronTaskHistoryId(cronTaskHistoryId);
                            taskLockRequest.setNode(ClientNode.UUID_STR);
                            TaskLockResponse taskLockResponse = client2ControlTransport.lockClient(taskLockRequest);
                            if (taskLockResponse == null || !taskLockResponse.isSuccess()) {
                                logLink("cannot get lock: "+ JSONAccessor.impl().format(taskLockRequest))
                                        .log(log::info).log(jobLog::log);
                                return;
                            }
                            //OK , start doing real taskSubmitRequest logic
                            JobEndpoint.JobContextImpl jobContext = new JobEndpoint.JobContextImpl(taskSubmitRequest, jobService);
                            jobContext.setCronTaskId(cronTaskId);
                            jobContext.setCronTaskHistoryId(cronTaskHistoryId);
                            jobContext.setIoThread(Thread.currentThread());
                            jobInnerContextMgr.add(cronTaskId,cronTaskHistoryId, jobContext);
                            try {
                                delayTaskExecutor.execute(jobContext, jobLog);
                            }
                            catch (Exception e) {
                                log.error(e.getMessage(), e);
                                jobLog.flush();
                                errorStackList(e).forEach(jobLog::log);
                                jobLog.flush();
                            }
                            finally {
                                jobInnerContextMgr.remove(cronTaskId,cronTaskHistoryId);
                                try {
                                    logLink(taskKey + " > completed: " + formatTime(LocalDateTime.now()))
                                            .log(log::info).log(jobLog::log);
                                    jobLog.flush();
                                } catch (Exception e) {
                                    log.error(e.getMessage(), e);
                                }
                            }

                        }
                        catch (Exception e) {
                            // 处理异常，避免容器崩溃
                            log.error("Error processing record from topic: {}", topic, e);
                            jobLog.flush();
                            errorStackList(e).forEach(jobLog::log);
                            jobLog.flush();
                        }
                        finally {
                            try {
                                jobLog.flush(true);
                            } catch (Exception e) {
                                log.error(e.getMessage(),e);
                            }
                            if (!autoCommit) {
                                acknowledgment.acknowledge();
                            }
                            MDC.remove("traceId");
                            MDC.remove("requestId");
                        }
                    }
                });

                ConcurrentMessageListenerContainer<String, String> container =
                        new ConcurrentMessageListenerContainer<>(consumerFactory, containerProps);
                // 可选：设置并发（分区数需支持）
                container.setConcurrency(Math.max(jobListenerRunnerKafkaConfig.getConcurrency(),1));

                // 可选：添加启动/停止监听器
                container.setApplicationEventPublisher(event -> {
                    // 可用于监控
                });

                container.start();
                containers.put(containerId, container);
                log.info("Started dynamic consumer for topic: "+topic+", group: "+groupId+", concurrency: "+container.getConcurrency());

                return container;
            }
        });
        return container!=null;
    }

    @Override
    public void close() {

        for (ConcurrentMessageListenerContainer<String, String> container : containers.values()) {
            try {
                container.stop();
            } catch (Exception e) {
                log.error(e.getMessage(),e);
            }
        }

    }


    @Override
    public void run(String... args) throws Exception {
        ShutdownManager.getOrCreate().register(JobListenerRunnerViaKafkaProvider.class.getName(), new Util.CatchRunnable() {
            @Override
            public void run() throws Exception {
                close();
            }
        });
        log.info("register shutdown hook, jobhub kafka listener");
    }
}
