package com.ovopark.jobhub.sdk.client.kafka;

import com.ovopark.jobhub.sdk.client.*;
import com.ovopark.jobhub.sdk.model.*;
import com.ovopark.kernel.shared.*;
import com.ovopark.kernel.shared.kv.KVEngine;
import com.ovopark.kernel.shared.vclient.ClientNode;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.kafka.clients.consumer.ConsumerConfig;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.common.TopicPartition;
import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.CommandLineRunner;
import org.springframework.kafka.core.ConsumerFactory;
import org.springframework.kafka.listener.AcknowledgingMessageListener;
import org.springframework.kafka.listener.ConcurrentMessageListenerContainer;
import org.springframework.kafka.listener.ConsumerSeekAware;
import org.springframework.kafka.listener.ContainerProperties;
import org.springframework.kafka.support.Acknowledgment;
import org.springframework.kafka.support.TopicPartitionOffset;
import org.springframework.stereotype.Component;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicLong;

import static com.ovopark.kernel.shared.Util.*;

@JobClientActive
@Component
@Slf4j
public class TaskListenerRunnerViaKafkaProvider implements JobService.TaskListenerRunnerProvider , CommandLineRunner {

    @Qualifier("com.ovopark.jobhub.sdk.client.kafka.KafkaConfig.consumerFactory")
    @Autowired
    ConsumerFactory<String, String> consumerFactory;

    private final Map<String, TopicPartitions> containers = new ConcurrentHashMap<>();

    @Autowired
    private JobService jobService;

    @Autowired
    private Client2ControlTransport client2ControlTransport;

    @Autowired
    private ClientNodeProvider clientNodeProvider;

    final static KVEngine.TtlFunc<String> ttlFunc=KVEngine.newTtl("t-p-heartbeat-ttl");

    final static AtomicLong vcc=new AtomicLong();

    @Override
    public String name() {
        return "kafka";
    }

    final ScheduledExecutorService scheduledExecutorService= Executors.newSingleThreadScheduledExecutor(newThreadFactory("create-kafka-consumer"));

    @Override
    public boolean start(String jobType, String beanUrl, String group, Long minVer
            , JobService.TaskListener taskListener,JobService.TaskListenerRunnerProviderConfig taskListenerRunnerProviderConfig) {
        boolean f = start0(jobType, beanUrl, group, minVer, taskListener, taskListenerRunnerProviderConfig);
        if (!f) {
            log.warn("cannot start consumer, schedule a delay task: "+beanUrl+":"+jobType+":"+group);
            scheduledExecutorService.schedule(new Runnable() {
                @Override
                public void run() {
                    try {
                        boolean f = start0(jobType, beanUrl, group, minVer, taskListener, taskListenerRunnerProviderConfig);
                        if (!f) {
                            scheduledExecutorService.schedule(this,10,TimeUnit.SECONDS);
                            log.warn("cannot start consumer, schedule a delay task: "+beanUrl+":"+jobType+":"+group);
                        }
                    }
                    catch (Exception e){
                        log.error(e.getMessage(),e);
                        scheduledExecutorService.schedule(this,10,TimeUnit.SECONDS);
                        log.warn("cannot start consumer, schedule a delay task: "+beanUrl+":"+jobType+":"+group);
                    }
                }
            },10,TimeUnit.SECONDS);
        }
        return f;
    }

    private boolean start0(final String jobType, final String beanUrl, final String group, Long minVer
            , JobService.TaskListener taskListener,JobService.TaskListenerRunnerProviderConfig taskListenerRunnerProviderConfig) {
        final String containerId=beanUrl+":"+jobType+":"+group;
        TopicPartitions topicPartitions = lock(containerId, new Callable<TopicPartitions>() {
            @Override
            public TopicPartitions call() throws Exception {
                TopicPartitions tps = containers.get(containerId);
                if (tps != null) {
                    return tps;
                }

                TaskMetaGetRequest taskMetaGetRequest = new TaskMetaGetRequest();
                taskMetaGetRequest.setType(jobType);
                taskMetaGetRequest.setGroup(group);
                TaskMetaGetResponse taskMetaGetResponse = jobService.taskMetaGet(taskMetaGetRequest);
                if (taskMetaGetResponse == null || isEmpty(taskMetaGetResponse.getTopic())) {
                    log.info("cannot find topic " + JSONAccessor.impl().format(taskMetaGetResponse)
                            + ", request: " + JSONAccessor.impl().format(taskMetaGetRequest));
                    return null;
                }
                final String topic = taskMetaGetResponse.getTopic();
                final String consumerGroup = isEmpty(taskListenerRunnerProviderConfig.getConsumerGroup()) ? containerId :
                        taskListenerRunnerProviderConfig.getConsumerGroup();
                // 创建容器
                ContainerProperties containerProps=null;
                String partitions = taskListenerRunnerProviderConfig.getPartitions();
                if (isNotEmpty(partitions)) {
                    List<TopicPartitionOffset> topicPartitionOffsetList=new ArrayList<>();
                    for (String p : partitions.split(",")) {
                        TopicPartitionOffset topicPartitionOffset=new TopicPartitionOffset(topic,Integer.parseInt(p));
                        topicPartitionOffsetList.add(topicPartitionOffset);

                    }
                    containerProps = new ContainerProperties(topicPartitionOffsetList.toArray(new TopicPartitionOffset[]{}));
                }
                else {
                    containerProps = new ContainerProperties(topic);
                }

                containerProps.setGroupId(consumerGroup);
                // 通过 consumer properties 覆盖 offset reset
                if (isNotEmpty(taskListenerRunnerProviderConfig.getOffsetReset())) {
                    containerProps.getKafkaConsumerProperties()
                            .put(ConsumerConfig.AUTO_OFFSET_RESET_CONFIG, taskListenerRunnerProviderConfig.getOffsetReset()); // e.g., "earliest"
                }

                final boolean autoCommit = taskListenerRunnerProviderConfig.isAutoCommit();
                containerProps.getKafkaConsumerProperties().put(ConsumerConfig.ENABLE_AUTO_COMMIT_CONFIG, autoCommit);
                if (!autoCommit) {
                    // 👇 关键：设置为 MANUAL 或 MANUAL_IMMEDIATE
                    containerProps.setAckMode(ContainerProperties.AckMode.MANUAL);
                }
                if (taskListenerRunnerProviderConfig.getPollIntervalTimeSec()>0) {
                    containerProps.getKafkaConsumerProperties().put(ConsumerConfig.MAX_POLL_INTERVAL_MS_CONFIG, taskListenerRunnerProviderConfig.getPollIntervalTimeSec()*1000);
                }

                if (taskListenerRunnerProviderConfig.getMaxPollRecords()>0) {
                    containerProps.getKafkaConsumerProperties().put(ConsumerConfig.MAX_POLL_RECORDS_CONFIG, taskListenerRunnerProviderConfig.getMaxPollRecords());
                }
                containerProps.getKafkaConsumerProperties().put(ConsumerConfig.PARTITION_ASSIGNMENT_STRATEGY_CONFIG
                        , "org.apache.kafka.clients.consumer.CooperativeStickyAssignor");

                Map<String, ScheduledThreadPoolExecutor> heartbeatThread = new ConcurrentHashMap<>();

                containerProps.setMessageListener(
                        new PartitionSeekOffsetMessageListener(jobType,taskListenerRunnerProviderConfig
                                ,containerId,consumerGroup,taskListener,heartbeatThread));

                TopicPartitions topicPartitions = new TopicPartitions();

                ConcurrentMessageListenerContainer<String, String> container =
                        new ConcurrentMessageListenerContainer<>(consumerFactory, containerProps);
                // 可选：设置并发（分区数需支持）
                container.setConcurrency(Math.max(taskListenerRunnerProviderConfig.getConcurrency(), 1));

                // 可选：添加启动/停止监听器
                container.setApplicationEventPublisher(event -> {
                    // 可用于监控
                });

                container.start();

                topicPartitions.setConcurrentMessageListenerContainer(container);
                topicPartitions.setHeartbeatThread(heartbeatThread);

                containers.put(containerId, topicPartitions);
                log.info("Started dynamic consumer for topic: " + topic + ", group: " + consumerGroup + ", concurrency: " + container.getConcurrency());

                return topicPartitions;
            }
        });
        return topicPartitions!=null;
    }


    class PartitionSeekOffsetMessageListener implements AcknowledgingMessageListener<String, String> , ConsumerSeekAware {

        final String jobType;

        final String containerId;

        final private String consumerGroup;

        final JobService.TaskListener taskListener;

        final boolean autoCommit;

        final Map<String, ScheduledThreadPoolExecutor> heartbeatThread;

        final private boolean ignoreTaskIfDone;

        final boolean remoteLogEnabled;

        public PartitionSeekOffsetMessageListener(String jobType
                , JobService.TaskListenerRunnerProviderConfig taskListenerRunnerProviderConfig
                , String containerId
                , String consumerGroup
                , JobService.TaskListener taskListener
                ,Map<String, ScheduledThreadPoolExecutor> heartbeatThread
        ) {
            this.jobType = jobType;
            this.containerId = containerId;
            this.consumerGroup = consumerGroup;
            this.taskListener = taskListener;
            this.autoCommit = taskListenerRunnerProviderConfig.isAutoCommit();
            this.heartbeatThread=heartbeatThread;
            this.ignoreTaskIfDone=taskListenerRunnerProviderConfig.isIgnoreTaskIfDone();
            this.remoteLogEnabled=taskListenerRunnerProviderConfig.isRemoteLogEnabled();
        }

        @Override
        public void onPartitionsRevoked(Collection<TopicPartition> partitions) {
            for (TopicPartition partition : partitions) {
                String t = partition.topic();
                int p = partition.partition();
                String key = key0(t, p);
                String k = RecordInfo.KEY_PREFIX + ":" + containerId + ":" + p;
                KVEngine.DeleteResult<String, Object> deleteResult = ttlFunc.delete(k);
                log.info("delete partition cache: "+k+", "+deleteResult.deleted());
                //
                ScheduledThreadPoolExecutor scheduledThreadPoolExecutor = heartbeatThread.get(key);
                try {
                    if (scheduledThreadPoolExecutor != null) {
                        scheduledThreadPoolExecutor.shutdown();
                    }
                    log.info("partition revoked, close heartbeat thread: " + t + ":" + p);
                } catch (Exception e) {
                    log.error(e.getMessage(), e);
                }
            }
        }

        private static String key0(String t, int p) {
            return t + ":" + p;
        }

        @Override
        public void onPartitionsAssigned(Map<TopicPartition, Long> assignments, ConsumerSeekCallback callback) {
            for (TopicPartition partition : assignments.keySet()) {
                String t = partition.topic();
                int p = partition.partition();
                String key = key0(t, p);

                //delete
                String k = RecordInfo.KEY_PREFIX + ":" + containerId + ":" + p;
                KVEngine.DeleteResult<String, Object> deleteResult = ttlFunc.delete(k);
                log.info("delete partition cache again: "+k+", "+deleteResult.deleted());
                //put current stat
                final RecordInfo ri = ttlFunc.putAndGet(k, (KVEngine.Upset<String, RecordInfo, RecordInfo>) getResult -> {
                    if (getResult.exists()) {
                        log.warn("error , partition cache should be empty");
                        throw new IllegalStateException("error , partition cache should be empty");
                    }
                    RecordInfo recordInfo = new RecordInfo();
                    recordInfo.setContainerId(containerId);
                    recordInfo.setPartition(p);
                    recordInfo.setTempUniqueId(uniqueFirstPart());
                    recordInfo.setProgressStat(new ProgressStat());
                    return recordInfo;
                }).value();

                ScheduledThreadPoolExecutor taskHeartbeat
                        = new ScheduledThreadPoolExecutor(
                        1, newThreadFactory("tmp"), new ThreadPoolExecutor.CallerRunsPolicy());
                taskHeartbeat.setKeepAliveTime(60, TimeUnit.SECONDS);
                taskHeartbeat.allowCoreThreadTimeOut(true);
                Runnable runnable = catchRunnable(new CatchRunnable() {
                    @Override
                    public void run() throws Exception {
                        //push topic heartbeat
                        String k = RecordInfo.KEY_PREFIX + ":" + containerId + ":" + p;
                        KVEngine.GetResult<String, Object> getResult = ttlFunc.get(k);
                        if (!getResult.exists()) {
                            return;
                        }

                        ClientNode clientNode = clientNodeProvider.clientNode();

                        RecordInfo recordInfo = (RecordInfo) getResult.value();
                        TaskGetResponse.Task task = recordInfo.getTask();
                        if (task==null) {
                            return;
                        }

                        TaskHeartbeatRequest taskHeartbeatRequest =new TaskHeartbeatRequest();
                        taskHeartbeatRequest.setTaskIdInES(task.getId());
                        taskHeartbeatRequest.setJobIdInES(task.getJobId());
                        taskHeartbeatRequest.setApp(clientNode.app());
                        taskHeartbeatRequest.setNode(ClientNode.UUID_STR);
                        taskHeartbeatRequest.setVcc(vcc.incrementAndGet());
                        //
                        taskHeartbeatRequest.setContainerId(containerId);
                        taskHeartbeatRequest.setPartition(ri.getPartition());
                        taskHeartbeatRequest.setConsumerGroup(consumerGroup);
                        taskHeartbeatRequest.setTempUniqueId(recordInfo.getTempUniqueId());
                        taskHeartbeatRequest.setTaskStartTimeStr(recordInfo.getTaskStartTimeStr());

                        taskHeartbeatRequest.setTask(task);
                        taskHeartbeatRequest.setHeartbeatData(recordInfo.getHeartbeatData());
                        taskHeartbeatRequest.setProgressStat(recordInfo.getProgressStat());
                        try {
                            log.info(k + ", heartbeat, request " + JSONAccessor.impl().format(taskHeartbeatRequest));
                            TaskHeartbeatResponse taskHeartbeatResponse = client2ControlTransport.heartbeatStat(taskHeartbeatRequest);
                            if (taskHeartbeatResponse ==null) {
                                recordInfo.setCancelledTaskIdInES(null);
                            }
                            else {
                                recordInfo.setCancelledTaskIdInES(taskHeartbeatResponse.getCancelledTaskIdInES());
                            }
                            log.info(k + ", heartbeat, response " + JSONAccessor.impl().format(taskHeartbeatResponse));
                        } catch (Exception e) {
                            log.error(e.getMessage(),e);
                        }
                    }
                });
                taskHeartbeat.scheduleWithFixedDelay(runnable, 0, 5, TimeUnit.SECONDS);
                heartbeatThread.put(key, taskHeartbeat);
                log.info("partition assigned,start heartbeat thread: " + t + ":" + p);
            }
        }

        @Override
        public void onMessage(ConsumerRecord<String, String> record, Acknowledgment acknowledgment) {
            String t = record.topic();
            int p = record.partition();
            //put current stat
            final RecordInfo recordInfo = ((RecordInfo) ttlFunc.get(RecordInfo.KEY_PREFIX + ":" + containerId + ":" + p).value());
            try {
                ProgressStat progressStat = recordInfo.getProgressStat();
                if (progressStat.getStartOffset()==0) {
                    progressStat.setStartOffset(record.offset());
                }
                progressStat.setTaskOffset(record.offset());
                //
                String value = record.value();
                TaskGetResponse.Task task = JSONAccessor.impl().read(value, TaskGetResponse.Task.class);
                recordInfo.setTask(task);
                recordInfo.setIoThread(Thread.currentThread());
                recordInfo.setTaskStartTimeStr(formatTime(LocalDateTime.now()));

                //
                if (ignoreTaskIfDone) {
                    TaskIfDoneRequest taskIfDoneRequest =new TaskIfDoneRequest();
                    taskIfDoneRequest.setTaskIdInES(task.getId());
                    taskIfDoneRequest.setConsumerGroup(consumerGroup);
                    TaskIfDoneResponse taskIfDoneResponse = client2ControlTransport.taskIfDone(taskIfDoneRequest);
                    if (taskIfDoneResponse !=null && taskIfDoneResponse.isDone()) {
                        log.info("cancel task: "+JSONAccessor.impl().format(taskIfDoneResponse));
                        return;
                    }
                }


                TaskUpdateRequest taskUpdateRequest = new TaskUpdateRequest();
                taskUpdateRequest.setId(task.getId());

                final String taskKey = "_job4kafka_" + task.getJobId() + "_task_" + task.getId();
                final JobLog jobLog = remoteLogEnabled? new JobLogImpl(client2ControlTransport, taskKey, 0)
                        :JobLog.noopIfNull(null);
                JobServiceImpl.TaskContextImpl taskContext = new JobServiceImpl.TaskContextImpl();
                taskContext.setTask(task);
                taskContext.setHeartbeatDataListener(recordInfo::setHeartbeatData);
                taskContext.setCancelledSupplier((ServiceProvider<Boolean>) () -> compare2(task.getId(),recordInfo.getCancelledTaskIdInES())==0);
                taskContext.setJobLog(jobLog);
                taskContext.setRecordInfoSupplier(()->recordInfo);

                recordInfo.setTaskContext(taskContext);
                //
                MDC.put("traceId", taskKey);
                MDC.put("requestId", taskKey);
                JobStatus jobStatus = JobStatus.COMPLETED;
                try {
                    taskListener.on(taskContext);
                    jobStatus = convert2Self(taskContext.getJobStatus(), JobStatus.COMPLETED);
                }
                catch (Exception e) {
                    log.error(e.getMessage(), e);
                    jobStatus = JobStatus.FAIL;
                    taskUpdateRequest.setCompletedDesc(e.getMessage());
                }
                finally {
                    if (!taskContext.isStatusManageManually()) {
                        try {
                            taskUpdateRequest.setId(task.getId());
                            taskUpdateRequest.setDocIndexName(task.getDocIndexName());

                            taskUpdateRequest.setStatus(jobStatus.name());
                            if (isNotEmpty(taskContext.getCompletedDesc())) {
                                taskUpdateRequest.setCompletedDesc(taskContext.getCompletedDesc());
                            }
                            taskUpdateRequest.setRequestDeviceUrl(taskContext.getRequestDeviceUrl());
                            taskUpdateRequest.setRequestDeviceArgs(taskContext.getRequestDeviceArgs());
                            taskUpdateRequest.setResponseFromDevice(taskContext.getResponseFromDevice());

                            taskUpdateRequest.setRetryCount(taskContext.task().getRetryCount());

                            TaskUpdateResponse taskUpdateResponse = jobService.taskUpdate(taskUpdateRequest);
                            log.info("taskUpdate: " + JSONAccessor.impl().format(taskUpdateResponse));

                        } catch (Exception e) {
                            log.error(e.getMessage(),e);
                        }
                    }

                    try {
                        List<String> contentList = taskContext.getContentList();
                        TaskLogPutRequest taskLogPutRequest = new TaskLogPutRequest();
                        taskLogPutRequest.setJobId(task.getJobId());
                        taskLogPutRequest.setTaskId(task.getId());
                        taskLogPutRequest.setType(jobType);
                        taskLogPutRequest.setContentList(contentList);

                        TaskLogPutResponse taskLogPutResponse = jobService.taskLog(taskLogPutRequest);
                        log.info("taskLog: " + JSONAccessor.impl().format(taskLogPutResponse));
                    } catch (Exception e) {
                        log.error(e.getMessage(),e);
                    }

                    try {
                        jobLog.flush(true);
                    } catch (Exception e) {
                        log.error(e.getMessage(),e);
                    }

                }
            }catch (Exception e) {
                // 处理异常，避免容器崩溃
                log.error("Error processing record from topic: {}", t, e);
            }
            finally {
                // clear current stat
                recordInfo.setIoThread(null);
                recordInfo.setTask(null);
                recordInfo.setTaskContext(null);
                if (!autoCommit) {
                    acknowledgment.acknowledge();
                }
                MDC.remove("traceId");
                MDC.remove("requestId");
            }
        }
    }

    @Override
    public void close() {

        for (TopicPartitions topicPartitions : containers.values()) {
            try {
                ConcurrentMessageListenerContainer<String, String> container = topicPartitions.getConcurrentMessageListenerContainer();
                container.stop();
            } catch (Exception e) {
                log.error(e.getMessage(),e);
            }
        }

    }


    @Override
    public void run(String... args) throws Exception {
        ShutdownManager.getOrCreate().register(TaskListenerRunnerViaKafkaProvider.class.getName(), new CatchRunnable() {
            @Override
            public void run() throws Exception {
                close();
            }
        });
        log.info("register shutdown hook, jobhub kafka listener");
    }

    RecordInfo get(String containerId,String p){
        String k = RecordInfo.KEY_PREFIX + ":" + containerId + ":" + p;
        KVEngine.GetResult<String, Object> getResult = ttlFunc.get(k);
        if (!getResult.exists()) {
            return null;
        }

        return (RecordInfo) getResult.value();
    }

    @Data
    public static class RecordInfo implements Model{

        static final String KEY_PREFIX="RecordInfo";

        private String topic;

        private int partition;

        private final long startTime=System.currentTimeMillis();

        private String containerId;

        private String tempUniqueId;

        // -------------

        transient private Thread ioThread;

        TaskGetResponse.Task task;

        transient JobService.TaskContext taskContext;

        private String taskStartTimeStr;

        //-------

        private Object heartbeatData;


        private String cancelledTaskIdInES;

        private ProgressStat progressStat;



    }

    @Data
    public static class ProgressStat implements Model{

        private long startOffset;

        private long taskOffset;

        private HeavyPerTopicPartitionTaskListener.SubProgressStat subProgressStat;

    }


    @Data
    static class TopicPartitions {

        ConcurrentMessageListenerContainer<String, String> concurrentMessageListenerContainer;

        private Map<String,ScheduledThreadPoolExecutor> heartbeatThread=new ConcurrentHashMap<>();

    }


}
