package com.ovopark.jobhub.sdk.client.kafka;

import com.ovopark.jobhub.sdk.client.JobService;
import com.ovopark.jobhub.sdk.model.TaskGetResponse;
import com.ovopark.kernel.shared.JSONAccessor;
import com.ovopark.kernel.shared.Model;
import com.ovopark.kernel.shared.OnlyPrivate;
import com.ovopark.kernel.shared.ServiceProvider;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerConfig;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.OffsetAndTimestamp;
import org.apache.kafka.common.PartitionInfo;
import org.apache.kafka.common.TopicPartition;
import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.kafka.core.ConsumerFactory;
import org.springframework.kafka.listener.AcknowledgingMessageListener;
import org.springframework.kafka.listener.ConsumerSeekAware;
import org.springframework.kafka.listener.ContainerProperties;
import org.springframework.kafka.listener.KafkaMessageListenerContainer;
import org.springframework.kafka.support.Acknowledgment;
import org.springframework.kafka.support.TopicPartitionOffset;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

import static com.ovopark.kernel.shared.Util.*;


@Slf4j
abstract public class HeavyPerTopicPartitionTaskListener implements JobService.TaskListener {


    @Autowired
    protected JobService jobService;


    @Override
    public void on(JobService.TaskContext taskContext) {

        return;

//        /*
//        JobService.TaskContext does not support async thread , important!!!
//         */
//
//        // if you do the task in other thread , and you need manage the status manually , configure it
//        // if you do not care the status , never configure
////        taskContext.statusManageManually();
//
//        TaskGetResponse.Task task = taskContext.task();
//        log.info("process task: "+JSONAccessor.impl().format(task));
//
//        // can pass task id
//        final String taskId = task.getId();
//
//        String jsonStr = task.getJsonStr();
//
//        if (isEmpty(jsonStr)) {
//            log.info("args is empty???"+ JSONAccessor.impl().format(task));
//            return;
//        }
//
//        SubKafkaDataListener subKafkaDataListener = subKafkaDataListener(taskContext);
//        subKafkaDataListener.beforeStartSubKafka(taskContext);
//        //
//        SubKafkaConsumerConfig subKafkaConsumerConfig = subKafkaConsumerConfig(taskContext);
//        final String topic = subKafkaConsumerConfig.topic();
//        final int partition = subKafkaConsumerConfig.partition();
//        final ConsumerFactory<?, ?> subKafkaConsumerFactory = subKafkaConsumerFactory();
//        if (subKafkaConsumerConfig.checkPartitionRange()) {
//            try (Consumer<?, ?> tempConsumer = subKafkaConsumerFactory.createConsumer()) {
//                List<PartitionInfo> partitions = tempConsumer.partitionsFor(topic);
//                if (isEmpty(partitions)) {
//                    log.info("topic does not exists?: "+topic);
//                    return;
//                }
//                if (partition >= 0 && partition >= partitions.size()) {
//                    log.info(topic+" , exceed max partitions?: "+partition+" > "+partitions.size());
//                    return;
//                }
//            }
//        }
//        // 创建容器
//        TopicPartitionOffset topicPartitionOffset=new TopicPartitionOffset(topic,partition);
//        ContainerProperties containerProps = new ContainerProperties(topicPartitionOffset);
//        containerProps.setGroupId(taskId+":"+uniqueFirstPart()+":"+partition);
//        // 通过 consumer properties 覆盖 offset reset
//        Map<String, Object> kafkaConsumerProperties = subKafkaConsumerConfig.kafkaConsumerProperties();
//        kafkaConsumerProperties.forEach(containerProps.getKafkaConsumerProperties()::put);
//        //partition.assignment.strategy:org.apache.kafka.clients.consumer.CooperativeStickyAssignor
//        containerProps.getKafkaConsumerProperties().put(ConsumerConfig.PARTITION_ASSIGNMENT_STRATEGY_CONFIG
//                , "org.apache.kafka.clients.consumer.CooperativeStickyAssignor");
//
//        CountDownLatch countDownLatch=new CountDownLatch(1);
//        SubKafkaContextImpl subKafkaContext= new SubKafkaContextImpl(topic, partition, countDownLatch);
//
//        containerProps.setMessageListener(new PerPartitionSeekOffsetMessageListener(
//                taskContext,subKafkaDataListener
//                ,subKafkaConsumerConfig
//                ,subKafkaConsumerFactory
//                ,subKafkaContext
//        ));
//        KafkaMessageListenerContainer<?, ?> container =
//                new KafkaMessageListenerContainer<>(subKafkaConsumerFactory, containerProps);
//
//        // 可选：添加启动/停止监听器
//        container.setApplicationEventPublisher(event -> {
//            // 可用于监控
//            log.info("event: "+event.getClass().getName());
//        });
//
//        container.start();
//        logLink("start kafka consumer: "+topic+":"+partition+", config: "
//                +JSONAccessor.impl().format(subKafkaConsumerConfig))
//                .log(log::info)
//                .log(taskContext.jobLog()::log)
//                .log(taskContext::appendLog);
//        try {
//            int expiredTimeMs = 60000;
//            for(;;){
//                if (countDownLatch.await(30, TimeUnit.SECONDS)
//                        || subKafkaContext.isCancelled()
//                        || taskContext.isCancelled()
//                        || System.currentTimeMillis()-subKafkaContext.latestTime.get()> expiredTimeMs
//                ) {
//                    break;
//                }
//            }
//
//            if (taskContext.isCancelled()) {
//                logLink("task is cancelled by remote, end")
//                        .log(log::info)
//                        .log(taskContext.jobLog()::log)
//                        .log(taskContext::appendLog);
//            }
//            if(subKafkaContext.isCancelled()){
//                logLink("task is cancelled by self, end")
//                        .log(log::info)
//                        .log(taskContext.jobLog()::log)
//                        .log(taskContext::appendLog);
//            }
//            if(System.currentTimeMillis()-subKafkaContext.latestTime.get()> expiredTimeMs){
//                //reset cancelled status
//                subKafkaContext.cancelled.set(true);
//                logLink("task is expired ("+formatTime(LocalDateTime.now())
//                        +" - "+ formatTime(dateTime(subKafkaContext.latestTime.get()))
//                        +" > "+expiredTimeMs+" ms ), end")
//                        .log(log::info)
//                        .log(taskContext.jobLog()::log)
//                        .log(taskContext::appendLog);
//            }
//
//        }
//        catch (Exception e) {
//            log.error(e.getMessage(),e);
//            throw convert2RuntimeException(e);
//        }
//        finally {
//            try {
//                container.stop();
//            } catch (Exception e) {
//                log.error(e.getMessage(),e);
//            }
//            try {
//                subKafkaDataListener.afterEndSubKafka(taskContext);
//            } catch (Exception e) {
//                log.error(e.getMessage(),e);
//            }
//            logLink("close kafka consumer: "+topic+":"+partition)
//                    .log(log::info)
//                    .log(taskContext.jobLog()::log)
//                    .log(taskContext::appendLog);
//        }
//
//        boolean running = container.isRunning();
//
//        //check on/off status... vThread???
//
//        // push log to jobhub server , as a task log record in ES , one record per call
//        // the method does not support async thread , important!!!
//        logLink("container is running?: "
//                +running+", consume data,  recordCount: "+subKafkaContext.recordCount.get()
//                +", skipRecordCount: "+subKafkaContext.skipRecordCount.get()
//        ).log(log::info).log(taskContext.jobLog()::log).log(taskContext::appendLog);
    }

    class PerPartitionSeekOffsetMessageListener implements AcknowledgingMessageListener<String, String> , ConsumerSeekAware{

        final JobService.TaskContext taskContext;

        final SubKafkaDataListener subKafkaDataListener;

        final SubKafkaConsumerConfig subKafkaConsumerConfig;

        final ConsumerFactory<?, ?> consumerFactory;

        final SubKafkaContextImpl subKafkaContext;


        public PerPartitionSeekOffsetMessageListener(JobService.TaskContext taskContext
                , SubKafkaDataListener subKafkaDataListener
                , SubKafkaConsumerConfig subKafkaConsumerConfig
                , ConsumerFactory<?, ?> consumerFactory
                , SubKafkaContextImpl subKafkaContext
        ) {
            this.taskContext = taskContext;
            this.subKafkaDataListener = subKafkaDataListener;
            this.subKafkaConsumerConfig = subKafkaConsumerConfig;
            this.consumerFactory=consumerFactory;
            this.subKafkaContext=subKafkaContext;
        }

        @Override
        public void onMessage(ConsumerRecord<String, String> record, Acknowledgment acknowledgment) {
            String topic = record.topic();
            Thread t=Thread.currentThread();
            TaskGetResponse.Task task=taskContext.task();
            String taskKey = "_job4kafka_" + task.getJobId() + "_task_" + task.getId();
            MDC.put("traceId", taskKey);
            MDC.put("requestId", taskKey);
            taskContext.captureSupplier(new ServiceProvider<Map<String, Object>>() {
                @Override
                public Map<String, Object> get() {

                    Map<String, Object> data=new HashMap<>();
                    data.put("subIoThread",t.getName()+"@"+t.hashCode());

                    StackTraceElement[] stackTraceElements = t.getStackTrace();
                    List<String> stackList=new ArrayList<>(stackTraceElements.length);
                    for (StackTraceElement stackTraceElement : stackTraceElements) {
                        stackList.add(stackTraceElement.toString());
                    }
                    data.put("subIoStack",stackList);
                    return data;
                }
            });

            TaskListenerRunnerViaKafkaProvider.KafkaTaskContext kafkaTaskContext =
                    (TaskListenerRunnerViaKafkaProvider.KafkaTaskContext) taskContext;
            SubProgressStat subProgressStat = kafkaTaskContext.getSubProgressStat();
            if (subProgressStat.getStartOffset()==0) {
                subProgressStat.setStartOffset(record.offset());
            }
            subProgressStat.setOffset(record.offset());
            try {
                subKafkaContext.incrementRecordCount();
                subKafkaContext.latestTime(System.currentTimeMillis());
                if (subKafkaContext.isCancelled()) {
                    subKafkaContext.incrementSkipRecordCount();
                    return;
                }
                //
                if (taskContext.isCancelled()) {
                    subKafkaContext.cancel();
                    subKafkaContext.incrementSkipRecordCount();
                    return;
                }
                long offset = record.offset();
                if (subKafkaContext.endOffset()>-1 && subKafkaContext.endOffset()<offset) {
                    subKafkaContext.cancel();
                    subKafkaContext.incrementSkipRecordCount();
                    log.info("exceed max offset: "+offset+" > "+subKafkaContext.endOffset());
                    return;
                }

                taskKey = "_job4kafka_" + task.getJobId() + "_task_" + task.getId()+":"+offset;
                MDC.put("traceId", taskKey);
                MDC.put("requestId", taskKey);
                subKafkaDataListener.onData(taskContext, new ConsumerRecordContextImpl(subKafkaContext) {
                    @Override
                    public ConsumerRecord<String, String> record() {
                        return record;
                    }
                });
            }
            catch (Exception e) {
                // 处理异常，避免容器崩溃
                log.error("Error processing record from topic: {}", topic, e);
            }
            finally {
                MDC.remove("traceId");
                MDC.remove("requestId");
            }
        }

        @Override
        public void onPartitionsAssigned(Map<TopicPartition, Long> assignments, ConsumerSeekCallback callback) {
            //
            TaskListenerRunnerViaKafkaProvider.KafkaTaskContext kafkaTaskContext =
                    (TaskListenerRunnerViaKafkaProvider.KafkaTaskContext) taskContext;
            SubProgressStat subProgressStat = kafkaTaskContext.getSubProgressStat();

            long startTimeMs = subKafkaConsumerConfig.startTimeMs();
            long endTimeMs = subKafkaConsumerConfig.endTimeMs();
            long defOffset = subKafkaConsumerConfig.offset();
            if (defOffset >0) {
                for (TopicPartition tp : assignments.keySet()) {
                    callback.seek(tp.topic(), tp.partition(), defOffset);
                    subKafkaContext.startOffset= defOffset;
                    subProgressStat.setStartOffset(defOffset);
                    log.info("seek offset ("+ defOffset +"), " +tp.topic()+":"+tp.partition());
                }
                log.info("cannot set end offset, use default time expired???");
                return;
            }

            if (startTimeMs>0 || endTimeMs>0) {
//                String bootstrap = (String) consumerFactory.getConfigurationProperties().get("bootstrap.servers");
//                Properties tempProps = new Properties();
//                tempProps.put(ConsumerConfig.BOOTSTRAP_SERVERS_CONFIG, bootstrap);
//                tempProps.put(ConsumerConfig.KEY_DESERIALIZER_CLASS_CONFIG, StringDeserializer.class);
//                tempProps.put(ConsumerConfig.VALUE_DESERIALIZER_CLASS_CONFIG, StringDeserializer.class);
                try (Consumer<?, ?> tempConsumer = consumerFactory.createConsumer()) {

                    if (startTimeMs>0) {
                        Map<TopicPartition, OffsetAndTimestamp> startOffsets = tempConsumer.offsetsForTimes(
                                assignments.keySet().stream()
                                        .collect(Collectors.toMap(tp -> tp, tp -> startTimeMs))
                        );
                        for (TopicPartition tp : assignments.keySet()) {
                            OffsetAndTimestamp ot = startOffsets.get(tp);
                            if (ot != null) {
                                callback.seek(tp.topic(), tp.partition(), ot.offset());
                                subKafkaContext.startOffset=ot.offset();
                                subProgressStat.setStartOffset(ot.offset());
                                log.info("seek offset ("+ot.offset()+") of timestamp ( "+startTimeMs+" ), "
                                        +tp.topic()+":"+tp.partition()
                                        );
                            } else {
                                callback.seekToEnd(tp.topic(), tp.partition());
                                log.info("seek offset (end) of timestamp ( "+startTimeMs+" ), "
                                        +tp.topic()+":"+tp.partition()
                                );
                            }
                        }
                    }

                    if (endTimeMs>0) {

                        Map<TopicPartition, Long> endedOffsets = tempConsumer.endOffsets(assignments.keySet());
                        endedOffsets.forEach((tp,v)->{
                            subKafkaContext.endOffset=v;
                            subProgressStat.setEndOffset(v);
                            log.info("first , set default latest end offset ("+v+"), " +tp.topic()+":"+tp.partition()
                            );
                        });

                        Map<TopicPartition, OffsetAndTimestamp> endOffsets = tempConsumer.offsetsForTimes(
                                assignments.keySet().stream()
                                        .collect(Collectors.toMap(tp -> tp, tp -> endTimeMs))
                        );
                        for (TopicPartition tp : assignments.keySet()) {
                            OffsetAndTimestamp ot = endOffsets.get(tp);
                            if (ot != null) {
                                subKafkaContext.endOffset=ot.offset();
                                subProgressStat.setEndOffset(ot.offset());
                                log.info("again ,set real end offset ("+ot.offset()+") of timestamp ( "+startTimeMs+" ), "
                                        +tp.topic()+":"+tp.partition()
                                );
                            }
                        }
                    }
                    else {
                        log.info("cannot set end offset, use default time expired???");
                    }

                }
            }

        }
    }


    public interface SubKafkaConsumerConfig extends Model {

        String topic();

        int partition();

        long offset();

        long startTimeMs();

        long endTimeMs();

        /**
         * @see ConsumerConfig#MAX_POLL_RECORDS_CONFIG
         * @see ConsumerConfig#MAX_PARTITION_FETCH_BYTES_CONFIG
         * @see ConsumerConfig#FETCH_MAX_BYTES_CONFIG
         * @see ConsumerConfig#FETCH_MAX_WAIT_MS_CONFIG
         * @return
         */
        Map<String,Object> kafkaConsumerProperties();

        default boolean checkPartitionRange(){return true;}

    }

    public interface ConsumerRecordContext extends SubKafkaContext{

        ConsumerRecord<String, String> record();

    }

    private abstract class ConsumerRecordContextImpl implements ConsumerRecordContext{

        final SubKafkaContext subKafkaContext;

        public ConsumerRecordContextImpl(SubKafkaContext subKafkaContext) {
            this.subKafkaContext = subKafkaContext;
        }

        @Override
        public String topic() {
            return subKafkaContext.topic();
        }

        @Override
        public int partition() {
            return subKafkaContext.partition();
        }

        @Override
        public void cancel() {
            subKafkaContext.cancel();
        }

        @Override
        public boolean isCancelled() {
            return subKafkaContext.isCancelled();
        }

        @Override
        public void incrementRecordCount() {
            subKafkaContext.incrementRecordCount();
        }

        @Override
        public void incrementSkipRecordCount() {
            subKafkaContext.incrementSkipRecordCount();
        }

        @Override
        public void latestTime(long latestTime) {
            subKafkaContext.latestTime(latestTime);
        }

    }

    @Data
    public static class SubProgressStat implements Model{

        private long offset;

        private long startOffset;

        private long endOffset;



    }

    public interface SubKafkaContext{

        String topic();

        int partition();

        void cancel();

        boolean isCancelled();

        @OnlyPrivate
        void incrementRecordCount();

        void incrementSkipRecordCount();
        
        void latestTime(long latestTime);

        default long startOffset(){return -1;}

        default long endOffset(){return -1;}

    }

    public interface SubKafkaDataListener {

        void beforeStartSubKafka(JobService.TaskContext taskContext);

        void onData(JobService.TaskContext taskContext,ConsumerRecordContext consumerRecordContext);

        void afterEndSubKafka(JobService.TaskContext taskContext);

    }

    abstract protected SubKafkaConsumerConfig subKafkaConsumerConfig(JobService.TaskContext taskContext);

    abstract protected ConsumerFactory<?, ?> subKafkaConsumerFactory();

    abstract protected SubKafkaDataListener subKafkaDataListener(JobService.TaskContext taskContext);

    private class SubKafkaContextImpl implements SubKafkaContext {

        private final String topic;
        private final int partition;
        private final AtomicBoolean cancelled;
        private final CountDownLatch countDownLatch;
        private final AtomicLong recordCount;
        private final AtomicLong skipRecordCount;
        private final AtomicLong latestTime;

        private long startOffset=-1;
        private long endOffset=-1;

        public SubKafkaContextImpl(String topic, int partition, CountDownLatch countDownLatch) {
            this.topic = topic;
            this.partition = partition;
            this.cancelled=new AtomicBoolean(false);
            this.countDownLatch = countDownLatch;
            this.recordCount=new AtomicLong();
            this.skipRecordCount=new AtomicLong();
            this.latestTime=new AtomicLong(System.currentTimeMillis());
        }

        @Override
        public String topic() {
            return topic;
        }

        @Override
        public int partition() {
            return partition;
        }

        @Override
        public void cancel() {
            cancelled.set(true);
            countDownLatch.countDown();
        }

        @Override
        public boolean isCancelled() {
            return cancelled.get();
        }

        @Override
        public void incrementRecordCount() {
            recordCount.incrementAndGet();
        }

        @Override
        public void incrementSkipRecordCount() {
            skipRecordCount.incrementAndGet();
        }

        @Override
        public void latestTime(long l) {
            latestTime.set(l);
        }

        @Override
        public long startOffset() {
            return startOffset;
        }

        @Override
        public long endOffset() {
            return endOffset;
        }
    }
}
