package com.ovopark.jobhub.sdk.client;

import com.ovopark.jobhub.sdk.model.internal.job.TaskLockRequest;
import com.ovopark.jobhub.sdk.model.internal.job.TaskLockResponse;
import com.ovopark.jobhub.sdk.model.internal.job.TaskSubmitRequest;
import com.ovopark.jobhub.sdk.model.internal.job.TaskSubmitResponse;
import com.ovopark.kernel.shared.JSONAccessor;
import com.ovopark.kernel.shared.vclient.ClientNode;
import com.ovopark.module.shared.BaseResult;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;

import java.time.LocalDateTime;
import java.util.Set;
import java.util.concurrent.*;

import static com.ovopark.kernel.shared.Util.*;

@JobClientActive
@Slf4j
@RestController("com.ovopark.jobhub.sdk.client.JobEndpoint")
@RequestMapping("/feign/jobhub-job/processing")
public class JobEndpoint {

    @Autowired
    private DelayTaskExecutor delayTaskExecutor;

    @Autowired
    private JobHubJobApi jobHubJobApi;

    @Autowired
    private JobService jobService;

    private static final ExecutorService jobExecutor = new ThreadPoolExecutor(
            Math.max(Math.min(Runtime.getRuntime().availableProcessors() * 2,64)
                    , Integer.parseInt(System.getProperty("JOBHUB_JOB_IO", "0")))
            , Math.max(Math.max(Runtime.getRuntime().availableProcessors() * 2,1024)
            , Integer.parseInt(System.getProperty("JOBHUB_JOB_IO", "0")))
            , 600, TimeUnit.SECONDS
            //Integer.MAX_VALUE ???
            , new LinkedBlockingQueue(1)
            , newThreadFactory("jobhub-job-io")
            , new ThreadPoolExecutor.AbortPolicy());

    final private Set<Long> executingCronTaskIdSet = ConcurrentHashMap.newKeySet();

    @RequestMapping("/submit")
    @ResponseBody
    public BaseResult<TaskSubmitResponse> submit(@RequestBody TaskSubmitRequest taskSubmitRequest){
        final String taskKey = "task_" + taskSubmitRequest.getCronTaskId() + "_" + taskSubmitRequest.getCronTaskHistoryId();
        MDC.put("traceId", taskKey);
        MDC.put("requestId", taskKey);
        log.info("receive a new task, we can do it???: "+JSONAccessor.impl().format(taskSubmitRequest));
        final TaskSubmitResponse taskSubmitResponse=new TaskSubmitResponse();
        taskSubmitResponse.setSubmitted(false);
        JobLog logBeforeSubmitted=new JobLogImpl(jobHubJobApi,taskKey,taskSubmitRequest.getCronTaskHistoryId());
        logLink("receive a new task, we can do it???: "+JSONAccessor.impl().format(taskSubmitRequest))
                .log(logBeforeSubmitted::log);
        try {
            lock("task:" + taskSubmitRequest.getCronTaskId(), new Callable<Void>() {
                @Override
                public Void call() throws Exception {

                    if (executingCronTaskIdSet.contains(taskSubmitRequest.getCronTaskId())) {
                        taskSubmitResponse.setSubmitted(false);
                        taskSubmitResponse.setDesc("Rejected,the previous task is executing");
                        logLink("Rejected,the previous task is executing")
                                .log(log::info).log(logBeforeSubmitted::log);
                        throw new IllegalStateException("the previous task is executing, reject the task: "
                                +JSONAccessor.impl().format(taskSubmitRequest));
                    }

                    TaskLockRequest taskLockRequest=new TaskLockRequest();
                    taskLockRequest.setCronTaskId(taskSubmitRequest.getCronTaskId());
                    taskLockRequest.setCronTaskHistoryId(taskSubmitRequest.getCronTaskHistoryId());
                    taskLockRequest.setNode(ClientNode.UUID_STR);
                    BaseResult<TaskLockResponse> baseResult = jobHubJobApi.lockClient(taskLockRequest);
                    if (baseResult == null || baseResult.getIsError()) {
                        logLink("cannot get lock: "+ JSONAccessor.impl().format(taskLockRequest))
                                .log(log::info).log(logBeforeSubmitted::log);
                        throw new IllegalStateException("cannot get lock: "+ JSONAccessor.impl().format(taskLockRequest));
                    }
                    TaskLockResponse taskLockResponse = baseResult.getData();
                    if (taskLockResponse == null || !taskLockResponse.isSuccess()) {
                        logLink("cannot get lock: "+ JSONAccessor.impl().format(taskLockRequest))
                                .log(log::info).log(logBeforeSubmitted::log);
                        throw new IllegalStateException("cannot get lock: "+ JSONAccessor.impl().format(taskLockRequest));
                    }

                    executingCronTaskIdSet.add(taskSubmitRequest.getCronTaskId());
                    try {
                        jobExecutor.execute(catchRunnable(() -> {
                            //OK , start doing real task logic
                            final JobLog jobLog=new JobLogImpl(jobHubJobApi,taskKey,taskSubmitRequest.getCronTaskHistoryId());
                            try {
                                delayTaskExecutor.execute(new JobContextImpl(taskSubmitRequest),jobLog);
                            }
                            catch (Exception e){
                                log.error(e.getMessage(),e);
                                jobLog.flush();
                                errorStackList(e).forEach(jobLog::log);
                                jobLog.flush();
                            }
                            finally {
                                executingCronTaskIdSet.remove(taskLockRequest.getCronTaskId());
                                try {
                                    logLink(taskKey+" > completed: "+formatTime(LocalDateTime.now()))
                                            .log(log::info).log(jobLog::log);
                                    jobLog.flush();
                                    jobLog.close();
                                }catch (Exception e){
                                    log.error(e.getMessage(),e);
                                }
                            }
                        }));
                        logLink("OK, the task is scheduled: "+taskKey)
                                .log(log::info).log(logBeforeSubmitted::log);
                        //OK ,  resource is available , executor accepts the task
                        taskSubmitResponse.setSubmitted(true);
                    } catch (RejectedExecutionException e) {
                        // resource is unavailable
                        executingCronTaskIdSet.remove(taskLockRequest.getCronTaskId());
                        taskSubmitResponse.setSubmitted(false);
                        taskSubmitResponse.setDesc("Rejected, resource is unavailable ");
                        logLink("Rejected, resource is unavailable ")
                                .log(log::info).log(logBeforeSubmitted::log);
                        throw convert2RuntimeException(e);
                    }
                    return null;
                }
            },15,TimeUnit.SECONDS);

        }
        catch (Exception e) {
            log.error(e.getMessage(),e);
            logBeforeSubmitted.flush();
            errorStackList(e).forEach(logBeforeSubmitted::log);
            logBeforeSubmitted.flush();
        }
        finally {
            try {
                logBeforeSubmitted.close();
            }
            catch (Exception e){
                log.error(e.getMessage(),e);
            }
            finally {
                MDC.remove("traceId");
                MDC.remove("requestId");
            }
        }
        return BaseResult.success(taskSubmitResponse);
    }


    @Data
    class JobContextImpl implements JobContext{

        final TaskSubmitRequest taskSubmitRequest;

        public JobContextImpl(TaskSubmitRequest taskSubmitRequest) {
            this.taskSubmitRequest = taskSubmitRequest;
        }

        @Override
        public String uri() {
            return taskSubmitRequest.getUri();
        }

        @Override
        public String args() {
            return taskSubmitRequest.getArgs();
        }

        @Override
        public String jobIdInES() {
            return taskSubmitRequest.getJobIdInES();
        }

        @Override
        public JobService jobService() {
            return jobService;
        }
    }


}

