package com.ovopark.jobhub.sdk.client;


import com.ovopark.jobhub.sdk.model.internal.ClientNodeRegisterRequest;
import com.ovopark.jobhub.sdk.model.internal.ClientNodeRegisterResponse;
import com.ovopark.jobhub.sdk.model.internal.TaskLockRequest;
import com.ovopark.jobhub.sdk.model.internal.TaskLockResponse;
import com.ovopark.kernel.shared.Config;
import com.ovopark.kernel.shared.JSONAccessor;
import com.ovopark.kernel.shared.vclient.Client2ServerTransport;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.io.FileSystemResource;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.ResponseEntity;
import org.springframework.http.client.SimpleClientHttpRequestFactory;
import org.springframework.http.converter.StringHttpMessageConverter;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestTemplate;

import java.nio.charset.StandardCharsets;
import java.util.Map;

@Slf4j
public class Client2ControlRestClient implements Client2ControlTransport {


    final boolean verbose= Config.ConfigPriority.option().getBoolean("JOBHUB_CLIENT_VERBOSE",false);

    private final RestTemplate restTemplate;

    final private ControlNode controlNode;

    public Client2ControlRestClient(ControlNode controlNode, RestTemplate restTemplate) {
        this.controlNode=controlNode;
        this.restTemplate=restTemplate;
    }

    public Client2ControlRestClient(ControlNode controlNode) {
        this.controlNode = controlNode;
        SimpleClientHttpRequestFactory factory = new SimpleClientHttpRequestFactory();
        factory.setReadTimeout(45_000);//ms
        factory.setConnectTimeout(15_000);//ms
        restTemplate = new RestTemplate(factory);
        restTemplate.getMessageConverters().add(0,new StringHttpMessageConverter(StandardCharsets.UTF_8));
    }



    @Override
    public ClientNodeRegisterResponse heartbeat(ClientNodeRegisterRequest clientNodeRegisterRequest) {
        return rpc(clientNodeRegisterRequest, "/feign/delay/cronJob/jobClient/heartbeat",ClientNodeRegisterResponse.class);
    }

    private <T> T rpc(Object request, String path,Class<T> clazz) {
        String url = "http://"+ controlNode.ip() + ":" + controlNode.port()+"/jobhub-control" ;
        url+=path;
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type","application/json");
        HttpEntity<String> formEntity = new HttpEntity<String>(JSONAccessor.impl().format(request), headers);
        try {
            ResponseEntity<String> responseEntity = restTemplate.postForEntity(url, formEntity, String.class);
            int statusCodeValue = responseEntity.getStatusCodeValue();
            if (statusCodeValue==200) {
                return JSONAccessor.impl().read(responseEntity.getBody(), clazz);
            }
            return null;
        } catch (Exception e) {
            if (verbose) {
                log.error(e.getMessage());
            }
            return null;
        }
    }

    private <T> T restWithFile(Map<String,Object> params, String file,  String path, Class<T> clazz){
        String url = "http://"+ controlNode.ip() + ":" + controlNode.port()+"/jobhub-control" ;
        url+=path;
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type","multipart/form-data");
        MultiValueMap<String, Object> multiValueMap = new LinkedMultiValueMap<>();
        for (Map.Entry<String, Object> entry : params.entrySet()) {
            multiValueMap.add(entry.getKey(),entry.getValue());
        }
        multiValueMap.add("file",new FileSystemResource(file));

        HttpEntity<MultiValueMap<String, Object>> formEntity = new HttpEntity<>(multiValueMap, headers);
        try {
            ResponseEntity<String> responseEntity = restTemplate.postForEntity(url, formEntity, String.class);
            int statusCodeValue = responseEntity.getStatusCodeValue();
            if (statusCodeValue==200) {
                return JSONAccessor.impl().read(responseEntity.getBody(), clazz);
            }
            return null;
        } catch (Exception e) {
            log.error(e.getMessage());
            return null;
        }

    }

    @Override
    public TaskLockResponse lockClient(TaskLockRequest taskLockRequest) {
        return rpc(taskLockRequest, "/feign/delay/cronJob/jobClient/lockClient",TaskLockResponse.class);
    }

    @Override
    public Client2ServerTransport.TaskLogResponse log(Client2ServerTransport.TaskLogRequest taskLogRequest) {
        return rpc(taskLogRequest, "/feign/delay/cronJob/jobClient/log", Client2ServerTransport.TaskLogResponse.class);
    }

}
