package com.ovopark.jobhub.sdk.client;

import com.ovopark.jobhub.sdk.model.*;
import com.ovopark.kernel.shared.Config;
import com.ovopark.kernel.shared.JSONAccessor;
import com.ovopark.kernel.shared.ServiceProvider;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Supplier;

import static com.ovopark.jobhub.sdk.client.JobLog.noopIfNull;
import static com.ovopark.kernel.shared.Util.*;

@Slf4j
@Component("com.ovopark.jobhub.sdk.client.JobServiceImpl")
public class JobServiceImpl implements JobService,ApplicationContextAware{

    @Autowired
    private JobHubJobApi jobHubJobApi;

    @Autowired(required = false)
    private SpringBeanUriGetter springBeanUriGetter;

    private List<TaskListenerRunnerProvider> taskListenerRunnerProviderList;

    private ApplicationContext applicationContext;

    final private static String useDefaultProvider = Config.ConfigPriority.option().getString("JOBHUB_TASK_PROVIDER","kafka");

    @Autowired(required = false)
    private JobListenerKafkaRunnerProvider jobListenerKafkaRunnerProvider;

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext=applicationContext;
    }

    @Autowired
    public void setTaskListenerRunnerProviderList(List<TaskListenerRunnerProvider> taskListenerRunnerProviderList) {
        this.taskListenerRunnerProviderList = taskListenerRunnerProviderList;
    }

    @Override
    public JobCreateResponse jobCreate(JobCreateRequest jobCreateRequest) {
        JobCreateResponse jobCreateResponse = jobHubJobApi.jobCreate(jobCreateRequest);
        return jobCreateResponse;
    }

    @Override
    public TaskCreateResponse taskCreate(TaskCreateRequest taskCreateRequest) {
        TaskCreateResponse taskCreateResponse = jobHubJobApi.taskCreate(taskCreateRequest);
        return taskCreateResponse;
    }

    @Override
    public TaskBulkCreateResponse taskBulkCreate(TaskBulkCreateRequest taskBulkCreateRequest) {
        TaskBulkCreateResponse taskBulkCreateResponse = jobHubJobApi.taskBulkCreate(taskBulkCreateRequest);
        return taskBulkCreateResponse;
    }

    @Override
    public TaskUpdateResponse taskUpdate(TaskUpdateRequest taskUpdateRequest) {
        TaskUpdateResponse taskUpdateResponse = jobHubJobApi.taskUpdate(taskUpdateRequest);
        return taskUpdateResponse;
    }

    @Override
    public TaskLogPutResponse taskLog(TaskLogPutRequest taskLogPutRequest) {
        TaskLogPutResponse taskLogPutResponse = jobHubJobApi.taskLog(taskLogPutRequest);
        return taskLogPutResponse;
    }

    @Override
    public TaskGetResponse taskGet(TaskGetRequest taskGetRequest) {
        TaskGetResponse taskGetResponse = jobHubJobApi.taskGet(taskGetRequest);
        return taskGetResponse;
    }

    @Override
    public TaskMetaGetResponse taskMetaGet(TaskMetaGetRequest taskMetaGetRequest) {
        try {
            TaskMetaGetResponse taskMetaGetResponse = jobHubJobApi.taskMetaGet(taskMetaGetRequest);
            return taskMetaGetResponse;
        }
        catch (Exception e){
            log.error(e.getMessage());
            return null;
        }
    }

    @Override
    public void doOnceOnDevice(Integer deviceStatusId, Integer deviceId, Integer userId, Integer groupId
            , String type, String jsonStr, String desc, String jobId, TaskListener taskListener) {
        doOnceOnDevice(deviceStatusId, deviceId, userId, groupId, type, jsonStr, desc, jobId, new TaskCallListener<Void>() {
            @Override
            public Void on(TaskContext taskContext) {
                taskListener.on(taskContext);
                return null;
            }
        });
    }

    @Override
    public <T> T doOnceOnDevice(Integer deviceStatusId, Integer deviceId, Integer userId, Integer groupId, String type, String jsonStr, String desc, String jobId, TaskCallListener<T> taskCallListener) {
        TaskCreateRequest taskCreateRequest=new TaskCreateRequest();
        taskCreateRequest.setDeviceStatusId(deviceStatusId);
        taskCreateRequest.setDeviceId(deviceId);
        taskCreateRequest.setUserId(userId);
        taskCreateRequest.setGroupId(groupId);
        taskCreateRequest.setType(type);
        taskCreateRequest.setJsonStr(jsonStr);
        taskCreateRequest.setDesc(desc);
        taskCreateRequest.setJobId(jobId);

        TaskCreateResponse taskCreateResponse = taskCreate(taskCreateRequest);
        log.info("taskCreate: "+ JSONAccessor.impl().format(taskCreateResponse));
        TaskContextImpl taskContext= new TaskContextImpl();
        JobStatus jobStatus=JobStatus.COMPLETED;
        TaskUpdateRequest taskUpdateRequest=new TaskUpdateRequest();
        try {
            T f = taskCallListener.on(taskContext);
            jobStatus=convert2Self(taskContext.jobStatus,JobStatus.COMPLETED);
            return f;
        }
        catch (Exception e) {
            jobStatus=convert2Self(taskContext.jobStatus,JobStatus.FAIL);
            log.error(e.getMessage(),e);
            throw convert2RuntimeException(e);
        }
        finally {
            if (!taskContext.isStatusManageManually()) {
                taskUpdateRequest.setId(taskCreateResponse.getId());
                taskUpdateRequest.setDocIndexName(taskCreateResponse.getDocIndexName());

                taskUpdateRequest.setStatus(jobStatus.name());
                taskUpdateRequest.setCompletedDesc(taskContext.getCompletedDesc());

                taskUpdateRequest.setRequestDeviceUrl(taskContext.getRequestDeviceUrl());
                taskUpdateRequest.setRequestDeviceArgs(taskContext.getRequestDeviceArgs());
                taskUpdateRequest.setResponseFromDevice(taskContext.getResponseFromDevice());

                TaskUpdateResponse taskUpdateResponse = taskUpdate(taskUpdateRequest);
                log.info("taskUpdate: "+ JSONAccessor.impl().format(taskUpdateResponse));
            }

            List<String> contentList = taskContext.getContentList();
            TaskLogPutRequest taskLogPutRequest =new TaskLogPutRequest();
            taskLogPutRequest.setDeviceStatusId(deviceStatusId);
            taskLogPutRequest.setDeviceId(deviceId);
            taskLogPutRequest.setJobId(jobId);
            taskLogPutRequest.setTaskId(taskCreateResponse.getId());
            taskLogPutRequest.setType(type);
            taskLogPutRequest.setContentList(contentList);

            TaskLogPutResponse taskLogPutResponse = taskLog(taskLogPutRequest);
            log.info("taskLog: "+ JSONAccessor.impl().format(taskLogPutResponse));

        }
    }

    @Override
    public Long cronTaskRegister(String name, String bean, String cron){
        final String uri = springBeanUriGetter.springBean(bean);
        return cronTaskRegister0(name, bean, cron, uri,null,null);
    }

    private Long cronTaskRegister0(String name, String bean,String cron, String uri
            ,JobListenerRunnerKafkaConfig jobListenerRunnerKafkaConfig,String kafkaTopic
    ) {
        try {
            CronTaskSaveRequest cronTaskSaveRequest=new CronTaskSaveRequest();
            cronTaskSaveRequest.setName(name);
            cronTaskSaveRequest.setLockName(true); // important!!!
            cronTaskSaveRequest.setCreateBy(-1);
            cronTaskSaveRequest.setModifyBy(-1);

            cronTaskSaveRequest.setUri(uri);
            cronTaskSaveRequest.setArgs(null);
            cronTaskSaveRequest.setCron(cron);
    //        cronTaskSaveRequest.setCron("0/10 * * * * ?");
            CronTaskSaveResponse cronTaskSaveResponse = jobHubJobApi.saveCronTask(cronTaskSaveRequest);
            log.info("create cron task result: "+ JSONAccessor.impl().format(cronTaskSaveResponse));

    //        if (baseResult==null || baseResult.getIsError()
    //                || baseResult.getData()==null || baseResult.getData().getId()==null) {
    //            throw new IllegalStateException("cannot create a new cron task.");
    //        }

            if (cronTaskSaveResponse!=null) {
                return cronTaskSaveResponse.getId();
            }

            return null;
        }
        catch (Exception e){
            log.error(e.getMessage(),e);
            return null;
        }
        finally {
            // start kafka client
            if (jobListenerRunnerKafkaConfig!=null) {
                jobListenerKafkaRunnerProvider.start(kafkaTopic,bean,jobListenerRunnerKafkaConfig);
            }
        }
    }

    @Override
    public Long cronTaskRegisterViaKafka(String name, String bean,String cron
            , String kafkaTopic, String partitionKey,JobListenerRunnerKafkaConfig jobListenerRunnerKafkaConfig) {
        final String uri = DelayTaskExecutor.kafkaChannel(kafkaTopic,partitionKey,bean);
        return cronTaskRegister0(name, bean, cron, uri,jobListenerRunnerKafkaConfig,kafkaTopic);
    }

    @Override
    public Long delayTaskRegister(String name, String bean, String args, long triggerTimeMs,
                                  int triggerIfMiss, Integer createBy) {
        final String uri = springBeanUriGetter.springBean(bean);
        return delayTaskRegister0(name, bean, args,triggerTimeMs, triggerIfMiss, createBy, uri,null,null);
    }

    private Long delayTaskRegister0(String name, String bean,String args, long triggerTimeMs,
                                    int triggerIfMiss, Integer createBy, String uri
            ,JobListenerRunnerKafkaConfig jobListenerRunnerKafkaConfig
                                    ,String kafkaTopic
    ) {
        DelayTaskSaveRequest cronTaskSaveRequest=new DelayTaskSaveRequest();
        cronTaskSaveRequest.setName(name);
        cronTaskSaveRequest.setCreateBy(createBy);
        cronTaskSaveRequest.setModifyBy(-1);

        cronTaskSaveRequest.setUri(uri);
        cronTaskSaveRequest.setArgs(args);
        cronTaskSaveRequest.setTriggerIfMiss(triggerIfMiss);
        cronTaskSaveRequest.setTriggerTimeMs(triggerTimeMs);

        try {
    //        cronTaskSaveRequest.setCron("0/10 * * * * ?");
            DelayTaskSaveResponse delayTaskSaveResponse = jobHubJobApi.saveDelayTask(cronTaskSaveRequest);
            log.info("create delay task result: "+ JSONAccessor.impl().format(delayTaskSaveResponse));

    //        if (baseResult==null || baseResult.getIsError()
    //                || baseResult.getData()==null || baseResult.getData().getId()==null) {
    //            throw new IllegalStateException("cannot create a new cron task.");
    //        }

            if (delayTaskSaveResponse!=null) {
                return delayTaskSaveResponse.getId();
            }

            return null;
        }
        catch (Exception e){
            log.error(e.getMessage(),e);
            return null;
        }
        finally {
            // start kafka client
            if (jobListenerRunnerKafkaConfig!=null) {
                jobListenerKafkaRunnerProvider.start(kafkaTopic,bean,jobListenerRunnerKafkaConfig);
            }
        }
    }

    @Override
    public Long delayTaskRegisterViaKafka(String name, String bean, String args
            , long triggerTimeMs, int triggerIfMiss, Integer createBy
            , String kafkaTopic, String partitionKey, JobListenerRunnerKafkaConfig jobListenerRunnerKafkaConfig) {
        final String uri = DelayTaskExecutor.kafkaChannel(kafkaTopic,partitionKey,bean);
        return delayTaskRegister0(name, bean,args, triggerTimeMs, triggerIfMiss, createBy, uri
                ,jobListenerRunnerKafkaConfig,kafkaTopic);
    }

    @Override
    public boolean cancelDelayTask(Long delayTaskId) {
        DelayTaskCancelRequest delayTaskCancelRequest=new DelayTaskCancelRequest();
        delayTaskCancelRequest.setId(delayTaskId);
        DelayTaskCancelResponse delayTaskCancelResponse = jobHubJobApi.cancelDelayTask(delayTaskCancelRequest);
        if (delayTaskCancelResponse==null) {
            return false;
        }
        return delayTaskCancelResponse.isSuccess();
    }

    @Override
    public TaskContext mockTaskContext(TaskGetResponse.Task task) {
        TaskContextImpl taskContext=new TaskContextImpl();
        taskContext.setTask(task);
        taskContext.setJobLog(noopIfNull(null));
        AtomicBoolean cancelled=new AtomicBoolean();
        taskContext.setHeartbeatDataListener(data -> {});
        taskContext.setCancelledSupplier((ServiceProvider<Boolean>) cancelled::get);
        return taskContext;
    }

    @Override
    public JobContext mockJobContext(String name, String uri, String args, String jobIdInES) {
        if (isEmpty(jobIdInES)) {
            //create new job
            JobCreateRequest jobCreateRequest=new JobCreateRequest();
            jobCreateRequest.setJsonStr(args);
            jobCreateRequest.setType("mockJobContext");
            jobCreateRequest.setDesc("mockJobContext");
            JobCreateResponse jobCreateResponse = jobCreate(jobCreateRequest);
            jobIdInES=jobCreateResponse.getId();
        }

        String finalJobIdInES = jobIdInES;
        return new JobContext() {
            @Override
            public Long jobId() {
                return 0L;
            }

            @Override
            public String name() {
                return name;
            }

            @Override
            public String uri() {
                return uri;
            }

            @Override
            public String args() {
                return args;
            }

            @Override
            public String jobIdInES() {
                return finalJobIdInES;
            }

            @Override
            public JobService jobService() {
                return JobServiceImpl.this;
            }

            @Override
            public boolean isCancelled() {
                return false;
            }

            @Override
            public void heartbeatData(Object data) {

            }
        };
    }

    @Override
    public boolean register(String jobType, String beanUrl,String group, Long minVer) {
        TaskListenerRunnerProviderConfig taskListenerRunnerProviderConfig=new TaskListenerRunnerProviderConfig();
        taskListenerRunnerProviderConfig.setConcurrency(1);
        return register(jobType, beanUrl, group, minVer,taskListenerRunnerProviderConfig);
    }

    @Override
    public boolean register(String jobType, String beanUrl, String group, Long minVer, TaskListenerRunnerProviderConfig taskListenerRunnerProviderConfig) {
        Object object = applicationContext.getBean(beanUrl);
        if (!(object instanceof TaskListener)) {
            throw new RuntimeException("bean must implement "+TaskListener.class.getName());
        }

        String realProvider= isNotEmpty(useDefaultProvider)? useDefaultProvider :taskListenerRunnerProviderConfig.getProvider();

        TaskListenerRunnerProvider found=null;
        for (TaskListenerRunnerProvider taskListenerRunnerProvider : taskListenerRunnerProviderList) {
            if (realProvider.equalsIgnoreCase(taskListenerRunnerProvider.name())) {
                found=taskListenerRunnerProvider;
                break;
            }
        }

        if (found==null) {
            throw new RuntimeException("cannot find provider, missing kafka dependency ???: jobhub-sdk-client-kafka ");
        }

        return found.start(jobType,beanUrl,group,minVer,((TaskListener) object),taskListenerRunnerProviderConfig);
    }

    @Data
    public static class TaskContextImpl implements TaskContext{

        public static final int MAX_CAPACITY = 100;

        JobStatus jobStatus;

        String completedDesc;

        List<String> contentList=new ArrayList<>(MAX_CAPACITY);

        String requestDeviceUrl;

        String requestDeviceArgs;

        String responseFromDevice;

        TaskGetResponse.Task task;

        boolean statusManageManually;

        HeartbeatDataListener heartbeatDataListener;

        Supplier<Boolean> cancelledSupplier;

        Supplier<Map<String,Object>> subCaptureSupplier;

        final AtomicBoolean finalCancelled=new AtomicBoolean(false);

        JobLog jobLog;

        @Override
        public void status(JobStatus jobStatus) {
            this.jobStatus=jobStatus;
        }

        @Override
        public void completedDesc(String completedDesc) {
            this.completedDesc=completedDesc;
        }

        @Override
        synchronized public void appendLog(String content) {
            if (contentList.size()>MAX_CAPACITY) {
                return;
            }
            contentList.add(content);
        }

        @Override
        public TaskGetResponse.Task task() {
            return task;
        }

        @Override
        public void statusManageManually() {
            statusManageManually=true;
        }

        @Override
        public synchronized void addRetryCount() {
            task.setRetryCount(Optional.ofNullable(task().getRetryCount()).orElse(0) +1);
        }

        @Override
        public boolean isCancelled() {
            if (finalCancelled.get()) {
                return true;
            }
            Boolean f = cancelledSupplier.get();
            if (f) {
                finalCancelled.set(true);
            }
            return f;
        }

        @Override
        public void heartbeatData(Object data) {
            heartbeatDataListener.onHeartbeatData(data);
        }

        @Override
        public JobLog jobLog() {
            return jobLog;
        }

        @Override
        public Map<String, Object> capture() {
            return subCaptureSupplier==null?new HashMap<>():subCaptureSupplier.get();
        }

        @Override
        public void captureSupplier(Supplier<Map<String, Object>> subCaptureSupplier) {
            this.subCaptureSupplier=subCaptureSupplier;
        }

        @Override
        public void requestDeviceUrl(String requestUrl) {
            this.requestDeviceUrl=requestUrl;
        }

        @Override
        public void requestDeviceArgs(String requestArgs) {
            this.requestDeviceArgs=requestArgs;
        }

        @Override
        public void responseFromDevice(String response) {
            this.responseFromDevice=response;
        }
    }

}
