/*
 * Decompiled with CFR 0.152.
 */
package com.ovopark.device.platform.utils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.MDC;

public class BatchConcurrentProcessor {
    public static final int DEFAULT_BATCH_SIZE = 1000;
    public static final int DEFAULT_TIMEOUT_SECONDS = 30;
    public static final int DEFAULT_MAX_CONCURRENCY = 5000;
    private static final int SMALL_DATASET_THRESHOLD = 5000;
    private static final int MEDIUM_DATASET_THRESHOLD = 50000;
    private static final int LARGE_DATASET_THRESHOLD = 200000;
    private static final int SMALL_DATASET_BATCH_SIZE = 500;
    private static final int MEDIUM_DATASET_BATCH_SIZE = 1000;
    private static final int LARGE_DATASET_BATCH_SIZE = 2000;
    private static final int MAX_BATCH_SIZE = 5000;
    private static final int MAX_CONCURRENCY_SMALL = 500;
    private static final int MAX_CONCURRENCY_MEDIUM = 2000;
    private static final int MAX_CONCURRENCY_LARGE = 5000;

    private BatchConcurrentProcessor() {
    }

    public static <K, V> Map<K, V> processBatchConcurrently(List<K> inputList, Function<List<K>, Map<K, V>> batchProcessor, Logger logger) {
        return BatchConcurrentProcessor.processBatchConcurrently(inputList, batchProcessor, logger, "batchProcessing");
    }

    public static <K, V> Map<K, V> processBatchConcurrently(List<K> inputList, Function<List<K>, Map<K, V>> batchProcessor, Logger logger, String methodName) {
        return BatchConcurrentProcessor.processBatchConcurrently(inputList, batchProcessor, logger, methodName, 1000);
    }

    public static <K, V> Map<K, V> processBatchConcurrently(List<K> inputList, Function<List<K>, Map<K, V>> batchProcessor, Logger logger, String methodName, int customBatchSize) {
        return BatchConcurrentProcessor.processBatchConcurrently(inputList, batchProcessor, logger, methodName, customBatchSize, 30);
    }

    public static <K, V> Map<K, V> processBatchConcurrently(List<K> inputList, Function<List<K>, Map<K, V>> batchProcessor, Logger logger, String methodName, int customBatchSize, int timeoutSeconds) {
        if (inputList == null || inputList.isEmpty()) {
            logger.info("{}: empty input list", (Object)methodName);
            return new HashMap();
        }
        long startTime = System.currentTimeMillis();
        int totalSize = inputList.size();
        if (totalSize <= 5000) {
            Map<K, V> result = BatchConcurrentProcessor.processDirectly(inputList, batchProcessor, logger, methodName, startTime);
            long totalElapsedTime = System.currentTimeMillis() - startTime;
            logger.info("{}: total processing time for {} items: {}ms", new Object[]{methodName, totalSize, totalElapsedTime});
            return result;
        }
        int batchSize = BatchConcurrentProcessor.calculateOptimalBatchSize(totalSize, customBatchSize);
        int maxConcurrency = BatchConcurrentProcessor.calculateOptimalConcurrency(totalSize);
        logger.info("{}: processing {} items with batchSize={}, maxConcurrency={}", new Object[]{methodName, totalSize, batchSize, maxConcurrency});
        Map<K, V> result = BatchConcurrentProcessor.processWithBatches(inputList, batchProcessor, logger, methodName, batchSize, maxConcurrency, timeoutSeconds, startTime);
        long totalElapsedTime = System.currentTimeMillis() - startTime;
        logger.info("{}: total processing time for {} items: {}ms", new Object[]{methodName, totalSize, totalElapsedTime});
        return result;
    }

    private static <K, V> Map<K, V> processDirectly(List<K> inputList, Function<List<K>, Map<K, V>> batchProcessor, Logger logger, String methodName, long startTime) {
        String threadName = Thread.currentThread().getName();
        boolean isVirtualThread = threadName.contains("VirtualThread") || threadName.contains("ForkJoinPool");
        logger.info("{}: starting direct processing (thread: {}, virtual: {}) for {} items", new Object[]{methodName, threadName, isVirtualThread, inputList.size()});
        try {
            double successRate;
            HashMap result = batchProcessor.apply(inputList);
            long elapsedTime = System.currentTimeMillis() - startTime;
            boolean success = result != null && !result.isEmpty();
            int resultSize = result != null ? result.size() : 0;
            double d = successRate = !inputList.isEmpty() ? (double)resultSize / (double)inputList.size() * 100.0 : 0.0;
            if (success) {
                logger.info("{}: direct processing completed successfully, size={}, resultSize={}, elapsedTime={}ms, successRate={}%", new Object[]{methodName, inputList.size(), resultSize, elapsedTime, successRate});
            } else {
                logger.warn("{}: direct processing returned empty/null result, size={}, elapsedTime={}ms, successRate={}%", new Object[]{methodName, inputList.size(), elapsedTime, successRate});
            }
            return result != null ? result : new HashMap();
        }
        catch (Exception e) {
            long elapsedTime = System.currentTimeMillis() - startTime;
            logger.error("{}: direct processing failed, size={}, elapsedTime={}ms, error={}", new Object[]{methodName, inputList.size(), elapsedTime, e.getMessage(), e});
            return new HashMap();
        }
    }

    private static <K, V> Map<K, V> processWithBatches(List<K> inputList, Function<List<K>, Map<K, V>> batchProcessor, Logger logger, String methodName, int batchSize, int maxConcurrency, int timeoutSeconds, long startTime) {
        List<List<K>> batches = BatchConcurrentProcessor.partitionList(inputList, batchSize);
        Semaphore semaphore = new Semaphore(maxConcurrency);
        ConcurrentHashMap result = new ConcurrentHashMap();
        logger.info("{}: starting batch processing - totalSize={}, batchSize={}, batchCount={}, maxConcurrency={}, timeout={}s", new Object[]{methodName, inputList.size(), batchSize, batches.size(), maxConcurrency, timeoutSeconds});
        Map mdcContext = MDC.getCopyOfContextMap();
        int totalBatchDataSize = 0;
        for (int i = 0; i < batches.size(); ++i) {
            int batchDataSize = batches.get(i).size();
            totalBatchDataSize += batchDataSize;
            if (i < 5 || i >= batches.size() - 5) {
                logger.debug("{}: batch {} data size = {}", new Object[]{methodName, i, batchDataSize});
                continue;
            }
            if (i != 5) continue;
            logger.debug("{}: ... (omitting {} batches in middle) ...", (Object)methodName, (Object)(batches.size() - 10));
        }
        try (ExecutorService executor = Executors.newVirtualThreadPerTaskExecutor();){
            logger.info("{}: virtual thread executor started, total batches to process: {}", (Object)methodName, (Object)batches.size());
            List<CompletableFuture<Map<K, V>>> futures = batches.stream().map(batch -> BatchConcurrentProcessor.createBatchTask(batch, batchProcessor, semaphore, executor, logger, methodName, mdcContext)).collect(Collectors.toList());
            logger.debug("{}: all batch tasks created, waiting for completion", (Object)methodName);
            boolean timeoutOccurred = BatchConcurrentProcessor.waitForCompletion(futures, logger, methodName, timeoutSeconds);
            BatchConcurrentProcessor.mergeResults(futures, result, logger, methodName, timeoutOccurred);
        }
        catch (Exception e) {
            logger.error("{}: virtual thread executor exception occurred, error={}", new Object[]{methodName, e.getMessage(), e});
        }
        long elapsedTime = System.currentTimeMillis() - startTime;
        int expectedSize = inputList.size();
        int actualSize = result.size();
        double completionRate = expectedSize > 0 ? (double)actualSize / (double)expectedSize * 100.0 : 0.0;
        logger.info("{}: batch processing finalized - totalSize={}, resultSize={}, completionRate={}%, elapsedTime={}ms, batchesProcessed={}", new Object[]{methodName, expectedSize, actualSize, completionRate, elapsedTime, batches.size()});
        if (completionRate < 95.0) {
            logger.warn("{}: low completion rate detected! expected={}, actual={}, rate={}%, this may indicate batch processing issues", new Object[]{methodName, expectedSize, actualSize, completionRate});
        }
        return result;
    }

    private static <K, V> CompletableFuture<Map<K, V>> createBatchTask(List<K> batch, Function<List<K>, Map<K, V>> batchProcessor, Semaphore semaphore, ExecutorService executor, Logger logger, String methodName, Map<String, String> mdcContext) {
        int batchSize = batch.size();
        int availablePermits = semaphore.availablePermits();
        long startTime = System.currentTimeMillis();
        return CompletableFuture.supplyAsync(() -> {
            Map map;
            boolean acquired;
            String threadName;
            block22: {
                long acquireStartTime;
                block21: {
                    if (mdcContext != null) {
                        MDC.setContextMap((Map)mdcContext);
                    }
                    threadName = Thread.currentThread().getName();
                    acquired = false;
                    acquireStartTime = System.currentTimeMillis();
                    if (semaphore.tryAcquire(5L, TimeUnit.SECONDS)) break block21;
                    long acquireTime = System.currentTimeMillis() - acquireStartTime;
                    logger.warn("{}: virtual thread [{}] failed to acquire semaphore after {}ms, batchSize={}, availablePermits={}", new Object[]{methodName, threadName, acquireTime, batchSize, availablePermits});
                    HashMap hashMap = new HashMap();
                    MDC.clear();
                    if (acquired) {
                        semaphore.release();
                        logger.debug("{}: virtual thread [{}] released semaphore, remainingPermits={}", new Object[]{methodName, threadName, semaphore.availablePermits()});
                    }
                    return hashMap;
                }
                acquired = true;
                long acquireTime = System.currentTimeMillis() - acquireStartTime;
                logger.debug("{}: virtual thread [{}] acquired semaphore after {}ms, batchSize={}, remainingPermits={}", new Object[]{methodName, threadName, acquireTime, batchSize, semaphore.availablePermits()});
                if (!Thread.currentThread().isInterrupted()) break block22;
                logger.warn("{}: virtual thread [{}] interrupted before batch processing, batchSize={}", new Object[]{methodName, threadName, batchSize});
                HashMap hashMap = new HashMap();
                MDC.clear();
                if (acquired) {
                    semaphore.release();
                    logger.debug("{}: virtual thread [{}] released semaphore, remainingPermits={}", new Object[]{methodName, threadName, semaphore.availablePermits()});
                }
                return hashMap;
            }
            try {
                long processingTime;
                Map result = null;
                boolean batchProcessorSuccess = false;
                String errorMessage = null;
                try {
                    logger.debug("{}: virtual thread [{}] starting batch processing, batchSize={}", new Object[]{methodName, threadName, batchSize});
                    result = (Map)batchProcessor.apply(batch);
                    batchProcessorSuccess = true;
                    processingTime = System.currentTimeMillis() - startTime;
                    int resultSize = result != null ? result.size() : 0;
                    logger.info("{}: virtual thread [{}] batch processing completed successfully, batchSize={}, resultSize={}, processingTime={}ms", new Object[]{methodName, threadName, batchSize, resultSize, processingTime});
                }
                catch (Exception e) {
                    batchProcessorSuccess = false;
                    errorMessage = e.getMessage();
                    long processingTime2 = System.currentTimeMillis() - startTime;
                    logger.error("{}: virtual thread [{}] batch processing failed, batchSize={}, processingTime={}ms, error={}", new Object[]{methodName, threadName, batchSize, processingTime2, errorMessage, e});
                }
                processingTime = System.currentTimeMillis() - startTime;
                if (batchProcessorSuccess && result != null && !result.isEmpty()) {
                    logger.debug("{}: virtual thread [{}] returning result, batchSize={}, resultSize={}, successRate=100%", new Object[]{methodName, threadName, batchSize, result.size()});
                } else if (result == null || result.isEmpty()) {
                    logger.warn("{}: virtual thread [{}] returned empty result, batchSize={}, error={}, successRate=0%", new Object[]{methodName, threadName, batchSize, errorMessage != null ? errorMessage : "unknown"});
                } else {
                    logger.warn("{}: virtual thread [{}] batch processing failed, batchSize={}, error={}, successRate=0%", new Object[]{methodName, threadName, batchSize, errorMessage});
                }
                map = result != null ? result : new HashMap();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                long interruptTime = System.currentTimeMillis() - startTime;
                logger.warn("{}: virtual thread [{}] was interrupted during batch processing, batchSize={}, interruptTime={}ms", new Object[]{methodName, threadName, batchSize, interruptTime});
                HashMap hashMap = new HashMap();
                MDC.clear();
                if (acquired) {
                    semaphore.release();
                    logger.debug("{}: virtual thread [{}] released semaphore, remainingPermits={}", new Object[]{methodName, threadName, semaphore.availablePermits()});
                }
                return hashMap;
            }
            catch (Exception e2) {
                long errorTime = System.currentTimeMillis() - startTime;
                logger.error("{}: virtual thread [{}] encountered unexpected error, batchSize={}, errorTime={}ms, error={}", new Object[]{methodName, threadName, batchSize, errorTime, e2.getMessage(), e2});
                HashMap hashMap = new HashMap();
                {
                    catch (Throwable throwable) {
                        MDC.clear();
                        if (acquired) {
                            semaphore.release();
                            logger.debug("{}: virtual thread [{}] released semaphore, remainingPermits={}", new Object[]{methodName, threadName, semaphore.availablePermits()});
                        }
                        throw throwable;
                    }
                }
                MDC.clear();
                if (acquired) {
                    semaphore.release();
                    logger.debug("{}: virtual thread [{}] released semaphore, remainingPermits={}", new Object[]{methodName, threadName, semaphore.availablePermits()});
                }
                return hashMap;
            }
            MDC.clear();
            if (acquired) {
                semaphore.release();
                logger.debug("{}: virtual thread [{}] released semaphore, remainingPermits={}", new Object[]{methodName, threadName, semaphore.availablePermits()});
            }
            return map;
        }, executor);
    }

    private static <K, V> boolean waitForCompletion(List<CompletableFuture<Map<K, V>>> futures, Logger logger, String methodName, int timeoutSeconds) {
        logger.debug("{}: waiting for {} batch tasks to complete with {}s timeout", new Object[]{methodName, futures.size(), timeoutSeconds});
        CompletableFuture<Void> allFutures = CompletableFuture.allOf(futures.toArray(new CompletableFuture[0]));
        long waitStartTime = System.currentTimeMillis();
        try {
            allFutures.get(timeoutSeconds, TimeUnit.SECONDS);
            long actualWaitTime = System.currentTimeMillis() - waitStartTime;
            logger.info("{}: all batch tasks completed successfully, waitTime={}ms, totalBatches={}", new Object[]{methodName, actualWaitTime, futures.size()});
            return false;
        }
        catch (TimeoutException e) {
            long timeoutTime = System.currentTimeMillis() - waitStartTime;
            long completedCount = futures.stream().filter(CompletableFuture::isDone).count();
            long failedCount = futures.stream().filter(f -> f.isCompletedExceptionally() || f.isCancelled()).count();
            long pendingCount = (long)futures.size() - completedCount - failedCount;
            logger.warn("{}: TIMEOUT after {}ms ({} seconds timeout), completed={}/{}, failed={}, pending={}", new Object[]{methodName, timeoutTime, timeoutSeconds, completedCount, futures.size(), failedCount, pendingCount});
            if (completedCount < (long)futures.size()) {
                logger.warn("{}: timeout caused {} batches to fail/pending, completionRate={}%", new Object[]{methodName, (long)futures.size() - completedCount, (double)completedCount / (double)futures.size() * 100.0});
            }
            BatchConcurrentProcessor.cancelPendingFutures(futures);
            return true;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            long interruptTime = System.currentTimeMillis() - waitStartTime;
            logger.error("{}: WAITING INTERRUPTED after {}ms, completed={}/{}", new Object[]{methodName, interruptTime, futures.stream().filter(CompletableFuture::isDone).count(), futures.size(), e});
            BatchConcurrentProcessor.cancelPendingFutures(futures);
            return false;
        }
        catch (ExecutionException e) {
            long errorTime = System.currentTimeMillis() - waitStartTime;
            logger.error("{}: EXECUTION ERROR after {}ms, error={}", new Object[]{methodName, errorTime, e.getMessage(), e});
            return false;
        }
    }

    private static <K, V> void cancelPendingFutures(List<CompletableFuture<Map<K, V>>> futures) {
        int cancelledCount = 0;
        int alreadyDoneCount = 0;
        int alreadyCancelledCount = 0;
        for (int i = 0; i < futures.size(); ++i) {
            CompletableFuture<Map<K, V>> future = futures.get(i);
            if (!future.isDone() && !future.isCancelled()) {
                boolean cancelled = future.cancel(true);
                if (cancelled && ++cancelledCount > 5 && cancelledCount < futures.size() - 2) continue;
            }
            if (future.isDone()) {
                ++alreadyDoneCount;
                continue;
            }
            if (!future.isCancelled()) continue;
            ++alreadyCancelledCount;
        }
    }

    private static <K, V> void mergeResults(List<CompletableFuture<Map<K, V>>> futures, Map<K, V> result, Logger logger, String methodName, boolean timeoutOccurred) {
        int totalBatches = futures.size();
        int successCount = 0;
        int errorCount = 0;
        int emptyResultCount = 0;
        int cancelledCount = 0;
        long totalMergedItems = 0L;
        logger.debug("{}: starting result merging for {} futures", (Object)methodName, (Object)totalBatches);
        for (int i = 0; i < futures.size(); ++i) {
            CompletableFuture<Map<K, V>> future = futures.get(i);
            if (future.isDone() && !future.isCancelled()) {
                try {
                    Map batchResult = future.getNow(null);
                    if (batchResult != null && !batchResult.isEmpty()) {
                        result.putAll(batchResult);
                        totalMergedItems += (long)batchResult.size();
                        if (++successCount > 3 && successCount < totalBatches - 2) continue;
                        logger.debug("{}: merged batch {} successfully, batchSize={}, totalMerged={}", new Object[]{methodName, i, batchResult.size(), totalMergedItems});
                        continue;
                    }
                    ++emptyResultCount;
                    logger.debug("{}: batch {} returned empty result", (Object)methodName, (Object)i);
                }
                catch (Exception e) {
                    if (++errorCount <= 3) {
                        logger.error("{}: failed to get batch result from future {}, error={}", new Object[]{methodName, i, e.getMessage(), e});
                        continue;
                    }
                    if (errorCount != 4) continue;
                    logger.warn("{}: additional batch errors occurred (suppressed further details)", (Object)methodName);
                }
                continue;
            }
            if (!future.isCancelled()) continue;
            ++cancelledCount;
            logger.debug("{}: batch {} was cancelled", (Object)methodName, (Object)i);
        }
        double successRate = (double)successCount / (double)totalBatches * 100.0;
        logger.info("{}: result merging completed - totalBatches={}, success={}, emptyResults={}, errors={}, cancelled={}, successRate={}%, mergedItems={}", new Object[]{methodName, totalBatches, successCount, emptyResultCount, errorCount, cancelledCount, successRate, totalMergedItems});
        if (timeoutOccurred) {
            logger.warn("{}: processing completed with timeout - successRate={}%, this may indicate performance issues", (Object)methodName, (Object)successRate);
        } else if (successRate < 100.0) {
            logger.warn("{}: partial success - successRate={}%, investigate failed batches", (Object)methodName, (Object)successRate);
        } else {
            logger.info("{}: all batches merged successfully - successRate=100%", (Object)methodName);
        }
    }

    private static int calculateOptimalBatchSize(int totalSize, int customBatchSize) {
        if (customBatchSize > 0) {
            return Math.min(customBatchSize, 5000);
        }
        if (totalSize <= 5000) {
            return 500;
        }
        if (totalSize <= 50000) {
            return 1000;
        }
        if (totalSize <= 200000) {
            return 2000;
        }
        return 5000;
    }

    private static int calculateOptimalConcurrency(int totalSize) {
        if (totalSize <= 5000) {
            return Math.min(500, totalSize);
        }
        if (totalSize <= 50000) {
            return 2000;
        }
        return 5000;
    }

    private static <T> List<List<T>> partitionList(List<T> list, int batchSize) {
        ArrayList<List<T>> batches = new ArrayList<List<T>>();
        for (int i = 0; i < list.size(); i += batchSize) {
            int end = Math.min(i + batchSize, list.size());
            batches.add(new ArrayList<T>(list.subList(i, end)));
        }
        return batches;
    }
}

