package com.ovopark.training.enhancer.filter;

import com.ovopark.training.enhancer.utils.EhContextUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.annotation.Order;
import org.springframework.http.server.ServletServerHttpRequest;
import org.springframework.util.StringUtils;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

@Slf4j
@Order(110)
public class EhLogFilter extends EhAbstractRequestLoggingFilter {

    @Override
    protected void beforeRequest(HttpServletRequest request, String message) {
        EhContextUtil.setStartTime(System.currentTimeMillis());
        log.info(wrap(message));
    }

    @Override
    protected void afterRequest(HttpServletRequest request, String message) {
        log.info(wrap(message));
        long cost = System.currentTimeMillis() - EhContextUtil.getStartTime();
        log.info("接口耗时 cost {}", cost);
    }

    public String createMessage(HttpServletRequest request, String prefix, String suffix) {
        StringBuilder msg = new StringBuilder();
        buildOne(msg, "uri", request.getRequestURI());
        buildOne(msg, "method", request.getMethod());

        if (isIncludeQueryString()) {
            String queryString = request.getQueryString();
            if (queryString != null) {
                buildOne(msg, "queryString", queryString);
            }
        }
        buildOne(msg, "userId", EhContextUtil.getUserId());

        buildOne(msg, "enterpriseId", EhContextUtil.getEnterpriseId());

        if (isIncludeClientInfo()) {
            String client = request.getRemoteAddr();
            if (StringUtils.hasLength(client)) {
                buildOne(msg, "client", client);
            }
            HttpSession session = request.getSession(false);
            if (session != null) {
                buildOne(msg, "session", session.getId());
            }
            String user = request.getRemoteUser();
            if (user != null) {
                buildOne(msg, "user", user);
            }
        }

        if (isIncludeHeaders()) {
            buildOne(msg, "headers", new ServletServerHttpRequest(request).getHeaders());
        }

        if (isIncludePayload()) {
            String payload = getMessagePayload(request);
            if (payload != null) {
                buildOne(msg, "payload", payload);
            }
        }

        return msg.toString();
    }

    private String wrap(String message) {
        return "ehRequest|" + message;
    }

    private void buildOne(StringBuilder sb, String key, Object value) {
        sb.append("[").append(key).append(":").append(value).append("]");
    }

    @Override
    protected int getMaxPayloadLength() {
        return 1000;
    }

    @Override
    protected boolean isIncludeQueryString() {
        return true;
    }

    @Override
    protected boolean isIncludeHeaders() {
        return true;
    }

    @Override
    protected boolean isIncludePayload() {
        return true;
    }

    @Override
    protected boolean isIncludeClientInfo() {
        return true;
    }
}
