package com.ovopark.training.enhancer.filter;

import com.ovopark.training.enhancer.utils.EhContextUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.annotation.Order;
import org.springframework.http.server.ServletServerHttpRequest;
import org.springframework.util.StringUtils;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

@Slf4j
@Order(110)
public class EhLogFilter extends EhAbstractRequestLoggingFilter {

    @Override
    protected void beforeRequest(HttpServletRequest request, String message) {
        EhContextUtil.setStartTime(System.currentTimeMillis());
        log.info("ehBeforeRequest|{}", message);
    }

    @Override
    protected void afterRequest(HttpServletRequest request, String message) {
        StringBuilder sb = new StringBuilder(message);
        long cost = System.currentTimeMillis() - EhContextUtil.getStartTime();
        buildOne(sb, "cost", cost);
        log.info("ehRequest|{}", sb);
    }

    @Override
    public String createMessage(HttpServletRequest request, String prefix, String suffix) {
        StringBuilder msg = new StringBuilder();
        msg.append("uri").append("=").append(request.getRequestURI());
        buildOne(msg, "method", request.getMethod());

        String param = "";
        if (isIncludeQueryString()) {
            String queryString = request.getQueryString();
            if (queryString != null) {
                param += queryString;
            }
        }
        buildOne(msg, "userId", EhContextUtil.getUserId());

        buildOne(msg, "enterpriseId", EhContextUtil.getEnterpriseId());
        if (isIncludeClientInfo()) {
            String client = request.getRemoteAddr();
            if (StringUtils.hasLength(client)) {
                buildOne(msg, "client", client);
            }
            HttpSession session = request.getSession(false);
            if (session != null) {
                buildOne(msg, "session", session.getId());
            }
            String user = request.getRemoteUser();
            if (user != null) {
                buildOne(msg, "user", user);
            }
        }

        if (isIncludeHeaders()) {
            buildOne(msg, "headers", new ServletServerHttpRequest(request).getHeaders());
        }

        if (isIncludePayload()) {
            String payload = getMessagePayload(request);
            if (payload != null) {
                param += payload;
            }
        }

        buildOne(msg, "param", param);
        return msg.toString();
    }

    private void buildOne(StringBuilder sb, String key, Object value) {
        sb.append("##").append(key).append("=").append(value);
    }

    @Override
    protected int getMaxPayloadLength() {
        return 3000;
    }
}
