package com.ovopark.training.enhancer.subject.strategyfactory;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author roy
 */
@Slf4j
public class HandlerFactory<E, T extends BaseStrategy<E>> implements InitializingBean, ApplicationContextAware {

    private static final String TAG = " handlerFactory ";

    private ApplicationContext applicationContext;

    /**
     * 泛型策略接口类型
     */
    private final Class<T> strategyInterfaceType;

    /**
     * java泛型只存在于编译期，无法通过例如T.class的方式在运行时获取其类信息
     * 因此利用构造函数传入具体的策略类型class对象为getBeansOfType()方法
     * 提供参数
     *
     * @param strategyInterfaceType 要传入的策略接口类型
     */
    public HandlerFactory(Class<T> strategyInterfaceType) {
        this.strategyInterfaceType = strategyInterfaceType;
    }

    /**
     * 策略实例容器
     */
    private Map<E, T> mStrategyMap;

    /**
     * 根据不同参数类型获取对应的接口实现类
     *
     * @param type 参数类型
     * @return 参数类型对应的接口实现类
     */
    public T getStrategy(E type) {
        return mStrategyMap.get(type);
    }

    public void register(T strategy) {
        T t = mStrategyMap.get(strategy.getType());
        if (t != null) {
            log.error("策略工厂中已存在类型为 {} 的策略", strategy.getType());
            throw new RuntimeException("策略工厂中已存在类型为 " + strategy.getType() + " 的策略");
        }
        mStrategyMap.put(strategy.getType(), strategy);
    }

    @Override
    public void afterPropertiesSet() {
        Map<String, T> beansOfType = applicationContext.getBeansOfType(strategyInterfaceType);
        log.info("{} 策略工厂初始化开始 beans {}", TAG, beansOfType);

        mStrategyMap = beansOfType.values().stream()
                .filter(strategy -> StringUtils.isNotEmpty(strategy.getType().toString()))
                .collect(Collectors.toMap(BaseStrategy::getType, Function.identity()));
        log.info("{} 策略工厂初始化结束 beansMap {}", TAG, mStrategyMap);
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}