package com.ovopark.training.enhancer.subject.gracefulshutdown;

import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.cloud.client.actuator.HasFeatures;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;

import com.ovopark.training.enhancer.subject.gracefulshutdown.shutdown.AutoServiceRegistrationShutdown;
import com.ovopark.training.enhancer.subject.gracefulshutdown.shutdown.KafkaShutdown;

import lombok.extern.slf4j.Slf4j;

/**
 * 优雅停机自动配置类
 * 主要实现:
 * 1. 服务下线 - 从注册中心摘除服务实例
 * 2. 消息消费停止 - 停止Kafka消费
 * 3. 请求处理完成 - 等待已接收请求处理完成
 * 4. 资源释放 - 释放各种连接资源
 */
@Configuration
@ConditionalOnProperty(
    prefix = "endpoints.shutdown.graceful",
    name = {"enabled"},
    havingValue = "true",
    matchIfMissing = true
)
@Slf4j
@EnableConfigurationProperties({GracefulShutdownProperties.class})
public class GracefulShutdownAutoConfiguration {

    public GracefulShutdownAutoConfiguration() {
    }

    @Bean
    public HasFeatures gracefulShutdownFeatures() {
        return HasFeatures.namedFeature("graceful shutdown", GracefulShutdownEndpoint.class);
    }

    @Bean
    public GracefulHealth gracefulHealth() {
        return new GracefulHealth();
    }

    @Bean
    @ConditionalOnMissingBean
    protected GracefulShutdownEndpoint gracefulShutdownEndpoint() {
        return new GracefulShutdownEndpoint();
    }

    /**
     * 服务注册相关的优雅下线处理
     * 主要用于从注册中心(如Eureka/Nacos)摘除服务实例
     */
    @Bean
    @ConditionalOnClass(
        name = {"org.springframework.cloud.client.serviceregistry.AutoServiceRegistration"}
    )
    public AutoServiceRegistrationShutdown autoServiceRegistrationShutdown() {
        return new AutoServiceRegistrationShutdown();
    }

    /**
     * Kafka消息消费优雅停止处理
     * 确保消息消费完成后再关闭
     */
    @Bean
    @ConditionalOnClass(
        name = {"org.springframework.kafka.config.KafkaListenerEndpointRegistry"}
    )
    public KafkaShutdown kafkaShutdown() {
        return new KafkaShutdown();
    }

    /**
     * 停机管理器
     * 按照优先级顺序执行各个组件的停机流程
     */
    @Bean
    public ShutdownManager shutdownManager(List<Shutdown> shutdowns) {
        shutdowns = shutdowns.stream()
                .sorted(Comparator.comparing(Ordered::getOrder))
                .peek(shutdown -> log.info("加载停机处理器: {}", shutdown.getClass()))
                .collect(Collectors.toList());
        return new ShutdownManager(shutdowns);
    }

}
